require 'dalli'
require 'pe_license'
require 'pe_license/status'

module LicenseHelpers
  UPDATE_MUTEX ||= Mutex.new

  def memcache_client
    @memcache_client ||= Dalli::Client.new('127.0.0.1:11211',
                                           :compress => true,
                                           :namespace => 'license_manager',
                                           :expires_in => cache_expiration)
  end

  def http
    return @http unless @http.nil?

    begin
      key_content = IO.read(settings.cert_key_path)
    rescue => e
      raise "Could not read file #{settings.cert_key_path}: #{e}"
    end

    begin
      cert_content = IO.read(settings.cert_path)
    rescue => e
      raise "Could not read file #{settings.cert_path}: #{e}"
    end

    @http = Net::HTTP.new(settings.puppetdb_host, settings.puppetdb_port)

    @http.ca_file = settings.ca_file_path
    @http.cert = OpenSSL::X509::Certificate.new(cert_content)
    @http.key = OpenSSL::PKey::RSA.new(key_content)

    @http.use_ssl = true
    @http.verify_mode = OpenSSL::SSL::VERIFY_PEER
    @http
  end

  def get_license
    license = memcache_client.get(:license)
    unless license
      begin
        license = PELicense.from_file('/etc/puppetlabs/license.key')
        memcache_client.set :license, license
      rescue PELicense::InvalidLicenseError
        license = PELicense.complimentary_license
      rescue
        halt 500, "API failure"
      end
    end
    license
  end

  def get_license_count
    get_license.nodes
  end

  # get_node_count will immediately return a number.
  # that number always comes from the cache
  # in order to ensure that the user isn't waiting 
  # for the result to come back from puppetdb.
  # To make sure that we have a relatively up to date
  # node count, we'll spawn a thread to get the number
  # from puppetdb and put it in the cache. However, we
  # only want one of these threads executing at a time,
  # so we have them protected with a mutex.
  # In the case of an empty cache, we'll actually wait
  # for the thread to complete.
  def get_node_count
    nodes = memcache_client.get :node_count
    if nodes.nil?
      should_wait = true
      logger.debug 'node_count not in cache, will attempt to join call-to-puppetdb thread'
    else
      logger.info "retreived node_count of #{nodes} from cache"
    end
    begin
      logger.debug 'spawning thread to get node_count from puppetdb'
      t = Thread.new do
        logger.debug 'attempting to acquire node_count update mutex'
        if UPDATE_MUTEX.try_lock
          begin
            logger.debug 'have lock, attempting to retrieve node_count from puppetdb'
            nodes = refresh_cache
          rescue => e
            logger.warn e.inspect
          ensure
            logger.debug "releasing node_count update mutex"
            UPDATE_MUTEX.unlock
          end
        end
      end
      t.join if should_wait
    rescue => e
      logger.warn e.inspect
    end
    nodes
  end

  def refresh_cache
    result = PELicense::Status.retrieve(http)
    logger.debug "retrieved node_count from puppetdb: #{result.inspect}"
    memcache_client.set :node_count, result.nodes
    logger.debug "set node_count in cache to #{result.nodes}"
    result.nodes
  end

  def title_text(node_count, license)
    node_diff = node_count - license.nodes
    description = ''
    if node_diff > 0
      description = 'unlicensed'
    else
      node_diff = - node_diff
      description = 'remaining'
    end
    "#{node_diff} #{description} node#{ node_diff == 1 ? '' : 's'}"
  end

  def contact_us_text(node_count, license)
    erb :contact_us, :locals => {:node_count => node_count, :license => license }
  end

  def license_footer(license)
    erb :footer, :locals => {:license => license, :days_exceeded => license.days_exceeded, :days_left => license.days_remaining }
  end

  def cache_expiration
    # 1 day in seconds
    60 * 60 * 24
  end
end
