# This profile sets up the Orchestrator service.
#
# @param certname [String] The name of the service SSL certificate.
# @param database_host [String] The hostname running PostgreSQL.
# @param database_port [Integer] The port that PostgreSQL is listening on.
# @param database_name [String] The name of the orchestrator database.
# @param database_user [String] The user account for the orchestrator database.
# @param database_password [String] The password for the user set by database_user.
# @param ssl_listen_address [String] The network interface used for SSL connections.
# @param ssl_listen_port [Boolean] The port used for SSL connections.
# @param pcp_listen_port [Boolean] The port used for PCP connections.
# @param master_host [String] The hostname of the Puppet Master.
# @param puppetdb_host [String] The hostname of the PuppetDB server.
# @param rbac_host [String] The hostname of the RBAC service.
# @param rbac_port [String] The port of the RBAC service.
# @param rbac_prefix [String] The URL prefix of the RBAC service.
# @param use_application_services [Boolean] Whether to enable the Orchestrator.
class puppet_enterprise::profile::orchestrator(
  $certname                 = $::clientcert,
  $database_host            = $puppet_enterprise::database_host,
  $database_port            = $puppet_enterprise::database_port,
  $database_name            = $puppet_enterprise::orchestrator_database_name,
  $database_user            = $puppet_enterprise::orchestrator_database_user,
  $database_password        = $puppet_enterprise::orchestrator_database_password,
  $database_properties      = $puppet_enterprise::database_properties,
  $ssl_listen_address       = $puppet_enterprise::params::ssl_address,
  $ssl_listen_port          = 8143,
  $master_host              = $puppet_enterprise::puppet_master_host,
  $puppetdb_host            = $puppet_enterprise::puppetdb_host,
  $puppetdb_port            = $puppet_enterprise::puppetdb_port,
  $classifier_host          = $puppet_enterprise::console_host,
  $classifier_port          = $puppet_enterprise::api_port,
  $classifier_prefix        = $puppet_enterprise::classifier_url_prefix,
  $rbac_host                = $puppet_enterprise::console_host,
  $rbac_port                = $puppet_enterprise::api_port,
  $rbac_prefix              = $puppet_enterprise::rbac_url_prefix,
  $use_application_services = $puppet_enterprise::use_application_services,
  $pcp_listen_port          = 8142,
  $pcp_accept_consumers     = $puppet_enterprise::params::pcp_broker_accept_consumers,
  $pcp_delivery_consumers   = $puppet_enterprise::params::pcp_broker_delivery_consumers,
  Integer $pcp_timeout      = 5,
  Hash $java_args                = $puppet_enterprise::params::orchestrator_java_args,
) {
  $container = 'orchestration-services'
  Package <| tag == 'pe-orchestrator-packages' |> {
    before => Puppet_enterprise::Trapperkeeper::Orchestrator[$container]
  }

  $client_certname = $puppet_enterprise::params::orchestrator_client_certname

  puppet_enterprise::trapperkeeper::orchestrator { $container:
    database_host            => $database_host,
    database_name            => $database_name,
    database_user            => $database_user,
    database_password        => $database_password,
    database_properties      => $database_properties,
    master_url               => "https://${master_host}:8140",
    puppetdb_url             => "https://${puppetdb_host}:${puppetdb_port}",
    classifier_url           => "https://${classifier_host}:${classifier_port}${classifier_prefix}",
    rbac_url                 => "https://${rbac_host}:${rbac_port}${rbac_prefix}",
    pcp_broker_url           => "wss://${certname}:${pcp_listen_port}/pcp",
    notify                   => Service["pe-${container}"],
    use_application_services => $use_application_services,
    pcp_timeout              => $pcp_timeout,
  }

  puppet_enterprise::trapperkeeper::pcp_broker { $container:
    accept_consumers   => $pcp_accept_consumers,
    delivery_consumers => $pcp_delivery_consumers,
  }

  @@puppet_enterprise::certs::puppetdb_whitelist_entry { "export: ${client_certname}-for-puppetdb-whitelist":
    certname => $client_certname,
  }

  @@puppet_enterprise::certs::rbac_whitelist_entry { "export: ${client_certname}-for-rbac-whitelist":
    certname => $client_certname,
  }

  puppet_enterprise::trapperkeeper::bootstrap_cfg { "${container} webrouting-service":
    container => $container,
    namespace => 'puppetlabs.trapperkeeper.services.webrouting.webrouting-service',
    service   => 'webrouting-service',
    notify    => Service["pe-${container}"],
  }

  $confdir = "/etc/puppetlabs/${container}/conf.d"
  file { "${confdir}/webserver.conf":
    ensure => present,
    owner  => "pe-${container}",
    group  => "pe-${container}",
    mode   => '0640',
  }

  Pe_hocon_setting {
    ensure => present,
    path   => "${confdir}/webserver.conf",
    notify => Service["pe-${container}"]
  }

  pe_hocon_setting { "${container}.webserver.pcp-broker.client-auth":
    setting => 'webserver.pcp-broker.client-auth',
    value   => 'want',
  }

  pe_hocon_setting { "${container}.webserver.pcp-broker.ssl-host":
    setting => 'webserver.pcp-broker.ssl-host',
    value   => $ssl_listen_address,
  }

  pe_hocon_setting { "${container}.webserver.pcp-broker.ssl-port":
    setting => 'webserver.pcp-broker.ssl-port',
    value   => $pcp_listen_port,
  }

  pe_hocon_setting { "${container}.webserver.pcp-broker.ssl-ca-cert":
    setting => 'webserver.pcp-broker.ssl-ca-cert',
    value   => $puppet_enterprise::params::localcacert,
  }

  pe_hocon_setting { "${container}.webserver.pcp-broker.ssl-cert":
    setting => 'webserver.pcp-broker.ssl-cert',
    value   => "/etc/puppetlabs/${container}/ssl/${certname}.cert.pem",
  }

  pe_hocon_setting { "${container}.webserver.pcp-broker.ssl-key":
    setting => 'webserver.pcp-broker.ssl-key',
    value   => "/etc/puppetlabs/${container}/ssl/${certname}.private_key.pem",
  }

  pe_hocon_setting{ "${container}.webserver.pcp-broker.ssl-crl-path":
    setting => 'webserver.pcp-broker.ssl-crl-path',
    value   => $puppet_enterprise::params::hostcrl,
  }

  pe_hocon_setting { "${container}.web-router-service.broker-service.websocket.route":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.pcp.broker.service/broker-service".websocket.route',
    value   => '/pcp',
  }

  pe_hocon_setting { "${container}.web-router-service.broker-service.websocket.server":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.pcp.broker.service/broker-service".websocket.server',
    value   => 'pcp-broker',
  }

  pe_hocon_setting { "${container}.web-router-service.broker-service.metrics.route":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.pcp.broker.service/broker-service".metrics.route',
    value   => '/',
  }

  pe_hocon_setting { "${container}.web-router-service.broker-service.metrics.server":
    path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
    setting => 'web-router-service."puppetlabs.pcp.broker.service/broker-service".metrics.server',
    value   => 'pcp-broker',
  }

  if $use_application_services {
    pe_hocon_setting { "${container}.webserver.orchestrator.client-auth":
      setting => 'webserver.orchestrator.client-auth',
      value   => 'none',
    }

    pe_hocon_setting { "${container}.webserver.orchestrator.default-server":
      setting => 'webserver.orchestrator.default-server',
      value   => true,
    }

    pe_hocon_setting { "${container}.webserver.orchestrator.ssl-host":
      setting => 'webserver.orchestrator.ssl-host',
      value   => $ssl_listen_address,
    }

    pe_hocon_setting { "${container}.webserver.orchestrator.ssl-port":
      setting => 'webserver.orchestrator.ssl-port',
      value   => $ssl_listen_port,
    }

    pe_hocon_setting { "${container}.webserver.orchestrator.ssl-ca-cert":
      setting => 'webserver.orchestrator.ssl-ca-cert',
      value   => $puppet_enterprise::params::localcacert,
    }

    pe_hocon_setting { "${container}.webserver.orchestrator.ssl-cert":
      setting => 'webserver.orchestrator.ssl-cert',
      value   => "/etc/puppetlabs/${container}/ssl/${certname}.cert.pem",
    }

    pe_hocon_setting { "${container}.webserver.orchestrator.ssl-key":
      setting => 'webserver.orchestrator.ssl-key',
      value   => "/etc/puppetlabs/${container}/ssl/${certname}.private_key.pem",
    }

    pe_hocon_setting{ "${container}.webserver.orchestrator.ssl-crl-path":
      setting => 'webserver.orchestrator.ssl-crl-path',
      value   => $puppet_enterprise::params::hostcrl,
    }

    pe_hocon_setting { "${container}.web-router-service.orchestrator-service":
      path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
      setting => 'web-router-service."puppetlabs.orchestrator.service/orchestrator-service"',
      value   => '/orchestrator/v1',
    }
  }
  else {
    pe_hocon_setting { "${container}.webserver.orchestrator":
      ensure  => absent,
      setting => "webserver.orchestrator",
    }
    pe_hocon_setting { "${container}.web-router-service.orchestrator-service":
      path    => "/etc/puppetlabs/${container}/conf.d/web-routes.conf",
      ensure  => absent,
      setting => 'web-router-service."puppetlabs.orchestrator.service/orchestrator-service"',
    }
  }

  file { "/etc/puppetlabs/${container}/ssl":
    ensure => directory,
    mode   => '0600',
    owner  => "pe-${container}",
    group  => "pe-${container}",
  }

  puppet_enterprise::certs { "pe-${container}":
    certname => $certname,
    cert_dir => "/etc/puppetlabs/${container}/ssl",
    notify   => Service["pe-${container}"],
    require  => File["/etc/puppetlabs/${container}/ssl"],
  }

  puppet_enterprise::certs { $client_certname:
    certname => $client_certname,
    cert_dir => "/etc/puppetlabs/${container}/ssl",
    owner    => "pe-${container}",
    group    => "pe-${container}",
    notify   => Service["pe-${container}"],
    require  => File["/etc/puppetlabs/${container}/ssl"],
  }

  puppet_enterprise::trapperkeeper::java_args { $container :
    java_args => $java_args,
  }

  service { "pe-${container}":
    ensure => true,
    enable => true,
  }
}
