#!/opt/puppetlabs/puppet/bin/ruby
$LOAD_PATH.unshift File.join(File.dirname(__FILE__), 'lib')

require 'sinatra/base'
require 'sinatra/r18n'
require 'sinatra/content_for'
require 'rack/commonlogger'
require 'r18n-rails-api'
require 'json'
require 'logger'
require 'installer'
require 'installer/interview'
require 'installer/subscribers'
require './helpers/infra_helper'

# This works around a known bug in Rack::CommonLogger, which is fixed in master
# but unreleased as yet.
class Logger
  alias :write :<<
end

class Installer::App < Sinatra::Base
  include InfraHelper

  configure :development do
    require 'sinatra/reloader'
    register Sinatra::Reloader

    set :event_logger, Logger.new(STDERR)
    set :http_logger, Logger.new(STDERR)
  end

  configure :production do
    set :event_logger, Logger.new("/var/log/puppetlabs/installer/installer.log", "daily")
    set :http_logger, Logger.new("/var/log/puppetlabs/installer/http.log", "daily")
  end

  use Rack::CommonLogger, http_logger
  event_logger.level = Logger::INFO
  event_logger.formatter = proc do |level, date, _name, msg|
    "[#{date.utc.strftime("%Y-%m-%d %H:%M:%S.%L %Z")}] #{level} #{msg}\n"
  end

  configure do
    helpers Sinatra::ContentFor
    register Sinatra::R18n

    set :erb, :trim => '-'
    set :interviews, {}
    set :phases, {}

    enable :sessions
    disable :dump_errors, :raise_errors, :show_exceptions

    # Configure r18n
    R18n::I18n.default = 'en-us'
    R18n::Filters.on(:named_variables)
    R18n.default_places { 'assets/i18n' }

    installer_config = ENV['INSTALLER_CONFIG'] || 'installer.conf'
    config = {}

    if installer_config && File.readable?(installer_config)
      File.open(installer_config) do |f|
        json = JSON.load(f)

        if json['working_directory']
          config[:workdir] = json['working_directory']
        end

        if json['puppet_enterprise_location']
          config[:pe_location] = json['puppet_enterprise_location']
        end
      end
    end

    if config[:workdir]
      log_dir = "#{config[:workdir]}/log"
      unless File.directory? log_dir
        Dir.mkdir log_dir
      end
      event_log = File.new("#{log_dir}/installer.log", "a+")
      http_log = File.new("#{log_dir}/http.log", "a+")
    else
      event_log = File.new("/var/log/puppetlabs/installer/installer.log", "a+")
      http_log = File.new("/var/log/puppetlabs/installer/http.log", "a+")
    end

    set :event_logger, Logger.new(event_log, "daily")
    event_logger.level = Logger::INFO
    event_logger.formatter = proc do |level, date, _name, msg|
      "[#{date.utc.strftime("%Y-%m-%d %H:%M:%S.%L %Z")}] #{level} #{msg}\n"
    end

    set :http_logger, Logger.new(http_log, "daily")
    use Rack::CommonLogger, http_logger

    set :config, config
  end

  helpers do
    def display_error_page(error=nil)
      if error.nil?
        error = t.error.unknown_error
        backtrace = []
      else
        backtrace = error.backtrace
      end

      erb :error, :locals => { error: error, backtrace: backtrace }
    end

    def h(text)
      Rack::Utils.escape_html(text)
    end

    def css(file, options = {})
      options = options.map { |pair| pair.join('=') }.join(' ')
      %Q{<link href="/css/#{file}.css" rel="stylesheet" #{options} />}
    end

    def js(file)
      %Q{<script src="/js/#{file}.js"></script>}
    end
  end

  not_found do
    message = t.error.page_not_found
    display_error_page(message)
  end

  error do
    e = request.env['sinatra.error']
    display_error_page(e)
  end

  before %r{^/(interview|answers|summary|validate|deploy|execute)/?} do
    redirect to('/') unless session[:interview_id]
    @interview = settings.interviews[session[:interview_id]]
    @infra = @interview.infra
  end

  get '/' do
    erb :index, :locals => { version: get_pe_major_minor_version, installer_hostname: request.host }
  end

  get '/type/?' do
    interview = Installer::Interview.new(settings.event_logger, request.host, settings.config.merge(r18n: t))
    session[:interview_id] = interview.id
    settings.interviews[interview.id] = interview
    erb :type, :locals => { step_num: 1, version: get_pe_major_minor_version }
  end

  get '/interview/:type/?' do
    @interview.type = params[:type]
    @interview.generate_questions_hash
    @interview.generate_question_dependencies

    locals = {
      sections: @interview.sections,
      rules: @interview.dependencies,
      infra: @infra,
      step_num: 2
    }

    erb :interview, :locals => locals
  end

  post '/interview/:type/?' do
    # Reset the planner and infra on the interview so we don't reuse cached
    # data/results
    @interview.reset

    infra = convert_params_to_infra(params)
    if settings.config[:workdir]
      save_answer_files_to_disk(params, infra, settings.config[:workdir])
    else
      # Defaults to pwd
      save_answer_files_to_disk(params, infra)
    end

    @interview.infra = infra
    settings.phases[session[:interview_id]] = []
    redirect to('/summary')
  end

  get '/answers/:host.answers' do
    host = params[:host]
    if @infra.hosts[host]

      headers "content-type" => "text/plain"
      locals = {
        answers: @infra.hosts[host].generate_answers,
        interview_type: @interview.type,
        version: get_pe_major_minor_version
      }
      erb :answers, :layout => false, :locals => locals
    else
      redirect to('/')
    end
  end

  get '/summary/?' do
    master = @infra.find_hosts_with_roles([:master]).first.get_role_by_name(:master)
    console = @infra.find_hosts_with_roles([:console]).first
    console = console.get_role_by_name(:console)
    puppetdb = @infra.find_hosts_with_roles([:puppetdb]).first.get_role_by_name(:puppetdb)
    postgres = @infra.find_hosts_with_roles([:postgres]).first
    postgres_role = postgres.get_role_by_name(:postgres) if postgres

    locals = {
      infra: @infra,
      interview_type: @interview.type,
      master: master,
      console: console,
      puppetdb: puppetdb,
      postgres: postgres_role,
      step_num: 3
    }
    erb :summary, :locals => locals
  end

  get '/validate/?' do
    phase = :pre_verify
    plan = []

    steps = @interview.planner.plan[phase]
    steps.each do |step|
      plan << { name: step.name, description: step.description }
    end

    locals = {
      infra: @infra,
      interview_type: @interview.type,
      phase: phase,
      plan: plan,
      step_num: 4
    }

    erb :validate, :locals => locals
  end

  get '/deploy/?' do
    phase = :deploy_pe
    plan = []

    steps = @interview.planner.plan[phase]
    steps.each do |step|
      plan << { name: step.name, description: step.description }
    end

    console_url = @infra.console.hostname
    locals = {
      infra: @infra,
      interview_type: @interview.type,
      phase: phase,
      console_url: console_url,
      version: get_pe_major_minor_version,
      plan: plan,
      step_num: 5
    }

    erb :deploy, :locals => locals
  end

  get '/execute/:phase/?', :provides => 'text/event-stream' do
    stream :keep_open do |out|
      begin
        phase = params[:phase].to_sym
        @interview.event_subscribers.add(out)
        out.callback { @interview.event_subscribers.remove(out) }

        if !settings.phases[session[:interview_id]].include?(phase)
          # Lock to prevent running the same phase multiple times
          settings.phases[session[:interview_id]] << phase

          @interview.executor.execute(@interview.planner.plan[phase])
          phase_complete_event = Installer::Event.new(:phase_complete, phase)
          @interview.event_subscribers.publish(phase_complete_event)

          # Remove lock
          settings.phases[session[:interview_id]].delete(phase)
        end
      rescue Exception => e
        error_event = Installer::Event.new(:phase_error, phase)
        @interview.event_subscribers.publish(error_event)
      end
    end
  end

  post '/cleanup/?' do
    status = params[:status]

    settings.event_logger.info("Shutting down after install with status #{status}")

    # We need to use a stream in order to a) send a non-empty response to the
    # client and proceed to perform an action, and b) not have our SystemExit
    # rescued.
    stream do |out|
      out.callback { exit status == 'success' }

      out.close
    end
  end
end
