require 'installer/errors'
require 'installer/infra'
require 'installer/database_service'
require 'json'
require 'pathname'

module InfraHelper
  def convert_params_to_infra(params)
    infra = Installer::Infra.new(params)

    verify_params(params)

    ['master_hostname', 'database_hostname', 'puppetdb_hostname', 'console_hostname'].each {|k|
      params[k] = params[k].downcase if params[k]
    }

    master_alt_names = params['master_alt_names'].gsub(/\s+/, '').split(',').reject(&:empty?)
    pe_managed = !params.key?('database_hostname')
    database_port = params['database_port'] || 5432
    database_hostname = params['database_hostname'] || params['puppetdb_hostname'] || params['master_hostname']

    puppetdb_db = Installer::DatabaseService.new(
      database_hostname,
      database_port,
      params['puppetdb_database_name'] || 'pe-puppetdb',
      params['puppetdb_database_username'] || 'pe-puppetdb',
      infra.get_password(params, 'puppetdb_database_password'),
      pe_managed
    )

    rbac_db = Installer::DatabaseService.new(
      database_hostname,
      database_port,
      params['rbac_database_name'] || 'pe-rbac',
      params['rbac_database_username'] || 'pe-rbac',
      infra.get_password(params, 'rbac_database_password'),
      pe_managed
    )

    classifier_db = Installer::DatabaseService.new(
      database_hostname,
      database_port,
      params['classifier_database_name'] || 'pe-classifier',
      params['classifier_database_username'] || 'pe-classifier',
      infra.get_password(params, 'classifier_database_password'),
      pe_managed
    )

    activity_db = Installer::DatabaseService.new(
      database_hostname,
      database_port,
      params['activity_database_name'] || 'pe-activity',
      params['activity_database_username'] || 'pe-activity',
      infra.get_password(params, 'activity_database_password'),
      pe_managed
    )

    orchestrator_db = Installer::DatabaseService.new(
      database_hostname,
      database_port,
      params['orchestrator_database_name'] || 'pe-orchestrator',
      params['orchestrator_database_username'] || 'pe-orchestrator',
      infra.get_password(params, 'orchestrator_database_password'),
      pe_managed
    )

    postgres_role = Installer::Role::Postgres.new(
      :password => infra.get_password(params, 'database_password'),
    )

    postgres_role.add_database('puppetdb', puppetdb_db)
    postgres_role.add_database('rbac', rbac_db)
    postgres_role.add_database('classifier', classifier_db)
    postgres_role.add_database('activity', activity_db)
    postgres_role.add_database('orchestrator', orchestrator_db)


    master_role = master_role_from_params(params, orchestrator_db)
    puppetdb_role = puppetdb_role_from_params(params, puppetdb_db, rbac_db, classifier_db, activity_db)
    console_role = console_role_from_params(params, puppetdb_db, rbac_db, classifier_db, activity_db, orchestrator_db)

    if params['type'] == 'monolithic'
      host = Installer::Host.new(params['master_hostname'], {
        alt_names: master_alt_names,
        username: params['master_host_username'],
        ssh_key_file: params['master_ssh_key_file'],
        ssh_key_passphrase: params['master_ssh_key_passphrase'],
        sudo_password: params['master_sudo_password'],
        localhost: params['master_install_location'] == 'local' })
      infra.add_host(host)

      host.assign_role(master_role)
      host.assign_role(puppetdb_role)
      host.assign_role(console_role)
      host.assign_role(postgres_role) if pe_managed
    else
      master = Installer::Host.new(params['master_hostname'], {
        alt_names: master_alt_names,
        username: params['master_host_username'],
        ssh_key_file: params['master_ssh_key_file'],
        ssh_key_passphrase: params['master_ssh_key_passphrase'],
        sudo_password: params['master_sudo_password'],
        localhost: params['master_install_location'] == 'local' })
      master.assign_role(master_role)

      pdb = Installer::Host.new(params['puppetdb_hostname'], {
        username: params['puppetdb_host_username'],
        ssh_key_file: params['puppetdb_ssh_key_file'],
        ssh_key_passphrase: params['puppetdb_ssh_key_passphrase'],
        sudo_password: params['puppetdb_sudo_password'] })
      pdb.assign_role(puppetdb_role)
      pdb.assign_role(postgres_role) if pe_managed

      console = Installer::Host.new(params['console_hostname'], {
        username: params['console_host_username'],
        ssh_key_file: params['console_ssh_key_file'],
        ssh_key_passphrase: params['console_ssh_key_passphrase'],
        sudo_password: params['console_sudo_password'] })
      console.assign_role(console_role)

      infra.add_host(master)
      infra.add_host(pdb)
      infra.add_host(console)
    end

    infra.hosts.values.each do |h|
      h.assign_role(Installer::Role::CloudProvisioner.new)
      h.assign_role(Installer::Role::Agent.new(
        :master_hostname => params['master_hostname'],
        :agent_certname => h.hostname
      ))
    end

    infra
  end

  def master_role_from_params(params, orchestrator_db)
    all_in_one = params['type'] == 'monolithic'
    Installer::Role::Master.new(
      :all_in_one => all_in_one,
      :cert_name => params['master_hostname'],
      :console_hostname => params['console_hostname'] || params['master_hostname'],
      :puppetdb_hostname => params['puppetdb_hostname'] || params['master_hostname'],
      :orchestrator_db => orchestrator_db,
      :check_for_updates => params['check_for_updates']
    )
  end

  def puppetdb_role_from_params(params, puppetdb_db, rbac_db, classifier_db, activity_db)
    Installer::Role::PuppetDB.new(
      :master_certname => params['master_hostname'],
      :console_certname => params['console_certname'] || params['master_hostname'],
      :puppetdb_db => puppetdb_db,
      :rbac_db => rbac_db,
      :classifier_db => classifier_db,
      :activity_db => activity_db
    )
  end

  def console_role_from_params(params, puppetdb_db, rbac_db, classifier_db, activity_db, orchestrator_db)
    master_hostname = params['master_hostname']

    if params['use_application_services'] == 'on'
      use_application_services = true
    else
      use_application_services = false
    end

    Installer::Role::Console.new(
      :password => params['console_password'],
      :puppetdb_db => puppetdb_db,
      :rbac_db => rbac_db,
      :classifier_db => classifier_db,
      :activity_db => activity_db,
      :orchestrator_db => orchestrator_db,
      :puppetdb_hostname => params['puppetdb_hostname'] || master_hostname,
      :master_hostname => master_hostname,
      :master_certname => master_hostname,
      :use_application_services => use_application_services
    )
  end

  def get_interview_type(infra)
    infra.hosts.length == 1 ? 'monolithic' : 'split'
  end

  def get_pe_major_version
    version = get_pe_full_version
    version[/^\d+/] || version
  end

  def get_pe_major_minor_version
    version = get_pe_full_version
    version[/^\d+\.\d+/] || version
  end

  def get_pe_major_minor_patch_version
    version = get_pe_full_version
    version[/^\d+\.\d+\.\d+/] || version
  end

  def get_pe_full_version
    path_to_version_file = "/opt/puppetlabs/puppet/share/installer/installer/VERSION"
    if File.readable?(path_to_version_file)
      File.read(path_to_version_file).strip
    else
      'latest'
    end
  end

  def verify_params(params)
    # Params that are required by all installs
    required_params = [
      'type',
      'master_hostname',
      'master_alt_names',
      'console_password',
    ]

    # Params only needed by a split install
    if params['type'] == 'split'
      required_params.push(
        'puppetdb_hostname',
        'console_hostname',
      )
    end

    missing = required_params.select { |req| params[req].nil? || params[req].strip.empty? }

    raise(Installer::Error, "Missing params: #{missing.join(', ')}") if missing.size > 0

    # Ensure there are three unique hosts for a split install
    if params['type'] == 'split'
      hostnames = [params['master_hostname'], params['console_hostname'], params['puppetdb_hostname']]
      if hostnames.uniq.size < 3
        raise(Installer::Error, "Split install requires three different hosts, but we were only given #{hostnames.uniq.join(', ')}")
      end
    end
  end

  # Saves the answer files to disk for a given infra
  def save_answer_files_to_disk(params, infra, workdir=Dir.getwd)
    views_dir = Pathname.new("#{Dir.getwd}/views")
    raise "Could not find 'views' directory in #{Dir.getwd}." unless views_dir.directory?

    Dir.chdir(workdir) do
      answer_dir = 'answers'
      Dir.mkdir(answer_dir) unless File.exists?(answer_dir)

      interview_type = get_interview_type(infra)
      raise "Interview type could not be determined." unless interview_type

      version = get_pe_major_minor_version
      raise "PE version could not be determined." unless version

      Dir.chdir(answer_dir) do
        begin
          original_umask = File.umask(077)
          infra.host_answers.each do |host, answers|
            File.open("#{host}.answers", 'w') do |f|
              # uses interview_type and version
              template = ERB.new(File.new("#{views_dir}/answers.erb").read)
              f.puts(template.result(binding))
            end
          end

          details = {
            'version' => get_pe_full_version,
          }

          params.each do |k,v|
            if k.include?('password') || k.include?('passphrase')
              details[k] = 'removed'
            else
              details[k] = v
            end
          end

          File.open("details.json", 'w') do |f|
            f.write(JSON.pretty_generate(details))
          end
        ensure
          File.umask(original_umask)
        end
      end
    end
  end
end
