require 'forwardable'
require 'tsort'
require 'set'

# Models a dependency graph
class Installer::DependencyGraph
  extend Forwardable
  include TSort
  include Enumerable

  def_delegators :@hash, :size, :empty?

  def initialize(hash={})
    @hash = {}
    hash.each { |node, dependents| add_node(node, dependents) }
  end

  # Each or nodes will return in topoligcal order.
  alias_method :each, :tsort_each
  alias_method :nodes, :tsort

  # Add a new node and its dependents to the graph.
  def add_node(node, dependents=[])
    @hash[node] ||= Set.new
    if dependents
      @hash[node].merge dependents
      dependents.each { |dependent| add_node(dependent) }
    end

    self
  end

  # Add new dependents of a node.
  def add_dependent(node, dependents)
    dependents = Array(dependents)
    add_node(node, dependents)

    self
  end

  # Make nodes dependent on dependency
  def add_dependency(nodes, dependency)
    nodes = Array(nodes)
    nodes.each do |node|
      add_node(dependency, [node])
    end

    self
  end

  def dependents(node)
    @hash[node]
  end

  # Merge dependency graph on to this dependency graph.
  def merge!(other)
    merge_dg(self, other)

    self
  end

  # Merge two dependency graphs, preserving originals.
  def merge(other)
    dg = self.class.new(self.to_hash)
    merge_dg(dg, other)

    dg
  end

  def to_hash
    @hash
  end

  private
  def merge_dg(dg1, dg2)
    dg2.each do |node|
      dg1.add_node(node, dg2.dependents(node))
    end

    dg1
  end

  def tsort_each_node(&block)
    @hash.each_key(&block)
  end

  def tsort_each_child(node, &block)
    dependents(node).each(&block)
  end
end
