require 'installer/host'
require 'installer/role/agent'
require 'installer/role/console'
require 'installer/role/master'
require 'installer/role/puppetdb'
require 'installer/role/postgres'
require 'installer/role/cloud_provisioner'
require 'installer/dependency_graph'

class Installer::Infra
  attr_reader :infra, :hosts, :params
  attr_accessor :installer_hostname

  def initialize(params = {})
    @params = params
    @hosts = {}
    @install_order = Installer::DependencyGraph.new
  end

  def add_host(host)
    @hosts[host.hostname] = host
    @install_order.add_node(host.hostname)
  end

  def find_hosts_with_roles(roles)
    hosts_with_roles = Set.new
    roles.each do |role|
      hosts.values.each do |host|
        hosts_with_roles << host if host.roles.map(&:name).include? role
      end
    end

    hosts_with_roles
  end

  def master
    hosts.values.find { |host| host.roles.map(&:name).include? :master }
  end

  def puppetdb
    hosts.values.find { |host| host.roles.map(&:name).include? :puppetdb }
  end

  def console
    hosts.values.find { |host| host.roles.map(&:name).include? :console }
  end

  def install_order
    unsatisfied = {}
    hosts.each do |hostname, host|
      requirements = host.requires_roles
      required_hosts = find_hosts_with_roles(requirements) if requirements
      if requirements.size > 0 && required_hosts.size > 0
        @install_order.add_node(hostname, required_hosts.map(&:hostname))
      elsif requirements.size > 0 && required_hosts.size == 0
        unsatisfied[hostname] = [requirements]
      end
    end

    raise "Could not determine install order because there were unsatisfied dependencies: #{unsatisfied}." if unsatisfied.size > 0
    @install_order.nodes
  end

  def puppet_managed_hosts
    hosts.collect do |hostname, host|
      if host.puppet_managed
        hostname
      end
    end.compact
  end

  def host_answers
    answers = {}
    hosts.each do |hostname, host|
      answers[hostname] = host.generate_answers
    end

    answers
  end

  def host_answers_hash
    answers = {}
    hosts.each do |hostname, host|
      answers[hostname] = host.answers_hash
    end

    answers
  end

  # Helper method to return a random password if one has not been provided
  def get_password(params, key)
    params[key] ||= SecureRandom.urlsafe_base64
  end

  def valid?
    # Determines if the infra is in a valid state.
    true
  end
end
