require 'json'

class Installer::Interview
  attr_reader :id, :questions, :sections, :dependencies, :executor, :event_subscribers, :installer_hostname
  attr_accessor :infra, :type

  def initialize(logger, installer_hostname, opts={})
    @id = SecureRandom.uuid
    @installer_hostname = installer_hostname
    @infra = Installer::Infra.new
    @infra.installer_hostname = installer_hostname
    @logger = logger
    @planner = nil
    @opts = opts
    @event_subscribers = Installer::Subscribers.new(logger)
    @executor = Installer::Executor.new(logger) do |event|
      @event_subscribers.publish(event)
    end
  end

  def planner
    return @planner if @planner
    unless @infra.params.empty?
      @planner = Installer::Planner.new(@infra, @logger, @opts)
    end
  end

  def reset
    @infra = Installer::Infra.new
    @planner = nil
  end

  def infra=(infra)
    infra.installer_hostname = @installer_hostname
    @infra = infra
  end

  # This will take a hash and traverse it for a list of questions. From those
  # questions it will then generate a dependency graph saying "question b" depends on "question a".
  # if the key is 'requires' then the question it is required must be answered
  # (currently thats a checkbox being checked). If the key is 'unless' then the checkbox must be
  # unchecked to show the questions.
  def generate_question_dependencies
    show_if = {}
    result = {}
    @sections.each do |title, info|
      result = info.inject({}) do |temp_hash, question|
        show_if.merge!(question['show-if']) if question['show-if']
        { "show-if" => show_if }
      end
    end
    @dependencies = result
  end

  def generate_questions_hash
    @questions = load_json('questions.json')
    question_types = load_json('sections.json')
    question_types.each do |type,sections|
      sections.each do |k,v|
        v.map! do |question_key|
          raise Installer::Error, "Question key not found: #{question_key}" unless questions[question_key]
          question = process_question(question_key, questions[question_key])
          { "id" => question_key }.merge(question)
        end
      end
    end
    @sections = question_types[@type].merge!(question_types['shared'])
  end

  def process_question(key, question)
    case key
    when 'master_install_location'
      # If we're non-root, we don't support local installation
      if Process.uid != 0
        question['default'] = 'remote'
        question['disabled'] = ['local']
      end
    else
      ; # Do nothing for now, perhaps add additional questions in the future.
    end

    question
  end

  def load_json(file_name)
    JSON.load(File.read(File.join(File.dirname(__FILE__), '..', '..', "models", "json",  file_name)))
  end
end
