require 'pathname'
require 'installer/infra'
require 'installer/planner/step'

class Installer::Planner
  # Compute a plan for transitioning to target_infra
  #
  # @param target_infra [Infra] infra to develop a plan to transition to
  # @param &update_block [Block] block that will be called with Commands in the plan have updates
  #   - TODO: Plan status object/hash format thing here
  #
  # @note WARN: Will not create plan for arbitrary infra transtions! That is targeted for a future version!
  #   - TODO: Algorithmically design plan to transition from arbitrary infra to another arbitrary infra.
  #   - currently assumes there is no PE install already configured
  #   - currently assumes SSH/OS is configured on all hosts provided
  def initialize(target_infra, logger, opts={})
    @log_dir = '/var/log/puppetlabs/installer'
    @work_dir = '/opt/puppetlabs/puppet/share/installer'
    @etc_dir = '/etc/puppetlabs/installer'

    raise 'No r18n available' unless opts[:r18n]

    if opts[:workdir]
      @local_log_dir = opts[:workdir] + '/log'
      @local_work_dir = opts[:workdir]
      @local_etc_dir = opts[:workdir] + '/etc'
      @local_answer_dir = @local_work_dir + '/answers'
    else
      @local_log_dir = @log_dir
      @local_work_dir = @work_dir
      @local_etc_dir = @etc_dir
      @local_answer_dir = @local_work_dir + '/answers'
    end

    @local_installer_dir = opts[:pe_location] || @local_work_dir + '/installer'
    @local_packages_dir = @local_installer_dir + "/packages"

    @target_infra = target_infra

    # Executor that will handle executing the Commands
    # Keeps track of state for the hosts, passing context between Commands and doing connection pooling.
    @executor = Installer::Executor.new(logger)

    # Commands required to verify that we can connect to the infra hosts
    # If any of the Commands fail, do not proceed with installation.
    @pre_verify = []

    # Commands required to complete the installation/upgrade/reconfig in order.
    # TODO: Decide if this objective will clean up temp files after installing?
    # If any of the Commands fail, PE is in a broken state.
    @install = []

    # Commands required to verify PE is working.
    @post_verify = []

    # Commands required to remove a PE installation
    @destroy = []

    @pre_verify << Installer::Planner::Step::CheckLocalEnvironment.new(@target_infra, r18n: opts[:r18n],
                    :installer_dir => @local_installer_dir,
                    :work_dir => @local_work_dir)
    @pre_verify << Installer::Planner::Step::CheckDNS.new(@target_infra, r18n: opts[:r18n])

    # Check SSH if there are non-localhost hosts that we need to connect to
    if @target_infra.hosts.values.any? { |h| !h.localhost }
      @pre_verify << Installer::Planner::Step::ConnectSSH.new(@target_infra, r18n: opts[:r18n])
      @install << Installer::Planner::Step::ConnectSSH.new(@target_infra, r18n: opts[:r18n])
      @destroy << Installer::Planner::Step::ConnectSSH.new(@target_infra, r18n: opts[:r18n])
    end

    @pre_verify << Installer::Planner::Step::CheckRoot.new(@target_infra, r18n: opts[:r18n])

    # Check that each node can resolve DNS to the nodes it needs to talk to.
    @pre_verify << Installer::Planner::Step::CheckCrossDNS.new(@target_infra, r18n: opts[:r18n])

    # TODO: Figure out/add more sanity checks.
    # Check there is enough CPU/RAM/etc for roles
    @pre_verify << Installer::Planner::Step::HardwareCheck.new(@target_infra, r18n: opts[:r18n])

    # Check that nodes are compatible with PE Version
    @pre_verify << Installer::Planner::Step::CompatibilityCheck.new(@target_infra,
                    :packages_dir => @local_packages_dir,
                    :r18n => opts[:r18n])

    # Check that the filesystem is not read only, full, or otherwise fubar.
    @pre_verify << Installer::Planner::Step::FilesystemCheck.new(@target_infra, r18n: opts[:r18n])

    # Check that packaging is sane and (warn if?) Puppet FOSS is installed
    #@sanity << Installer::Planner::Step::PackageCheck.new(@target_infra)

    # Check that all ports that need to be opened are open
    #@sanity << Installer::Planner::Step::FirewallCheck.new(@target_infra)

    # Check for internet access??? Optional?
    # Check if PE ports are in use
    #@sanity << Installer::Planner::Step::NetworkCheck.new(@target_infra)

    @target_infra.install_order.each do |host|
      @install << Installer::Planner::Step::InstallPE.new(@target_infra,
                    :host => host,
                    :local_installer_dir => @local_installer_dir,
                    :log_dir => @log_dir,
                    :etc_dir => @etc_dir,
                    :local_answer_dir => @local_answer_dir, r18n: opts[:r18n])
    end

    # Commands required to verify PE was installed correctly.
    # If these Commands fail, PE is not fully functional and support should probably be contacted unless we have a known remediation?

    # Puppet test run to run modules, etc
    @target_infra.install_order.each do |host|
      @post_verify << Installer::Planner::Step::VerifyPuppet.new(@target_infra, :host => host, r18n: opts[:r18n])
    end

    @post_verify << Installer::Planner::Step::VerifyMCO.new(@target_infra, r18n: opts[:r18n])

    # Save log files to nodes
    @post_verify << Installer::Planner::Step::BackupLogs.new(@target_infra,
                    :local_log_dir => @local_log_dir,
                    :log_dir => @log_dir,
                    :etc_dir => @etc_dir,
                    :local_answer_dir => @local_answer_dir, r18n: opts[:r18n])

    @target_infra.hosts.each do |hostname, host|
      @destroy << Installer::Planner::Step::PurgePE.new(@target_infra, :host => hostname,
                    :local_installer_dir => @local_installer_dir, r18n: opts[:r18n])
      @destroy << Installer::Planner::Step::DestroyPE.new(@target_infra, :host => hostname, r18n: opts[:r18n])
    end
  end

  def plan
    {
      :pre_verify => @pre_verify,
      :deploy_pe => @install + @post_verify,
      :install => @install,
      :post_verify => @post_verify,
      :destroy => @destroy,
    }
  end
end
