require 'installer/action'
require 'installer/errors'

class Installer::Planner::Step::BackupLogs < Installer::Planner::Step
  def initialize(infra, data={}, pre=nil, post=nil)
    super

    @stop_on_fail = true

    raise 'Undefined local logging directory' unless data[:local_log_dir]
    @local_log_dir = data[:local_log_dir]

    raise 'Undefined local answers directory' unless data[:local_answer_dir]
    @local_answer_dir = data[:local_answer_dir]

    raise 'Undefined remote configuration directory' unless data[:etc_dir]
    @etc_dir = data[:etc_dir]

    raise 'Undefined local logging directory' unless data[:log_dir]
    @log_dir = data[:log_dir]
  end

  # Call from executor with a block to handle the actions.
  def perform_step(&execution_block)
    @infra.hosts.each do |hostname, host|
      # Create temp log dir
      mktempdir = Installer::Action::Mktemp.new(host, "pe-installer-log-XXXXX")
      yield mktempdir
      unless mktempdir.success?
        error(hostname, @t.step.backup_logs.creating_tmp_failed(error: mktempdir.error_message(hostname)))
        next
      end

      tmpdir = mktempdir.dir

      # Copy log file
      local_installer_log = "#{@local_log_dir}/installer.log"
      tmp_installer_log = "#{tmpdir}/installer.log"
      copylogs = Installer::Action::Upload.new(:to_host => host, :content => File.read(local_installer_log), :to => tmp_installer_log)
      yield copylogs
      unless copylogs.success?
        error(hostname, @t.step.backup_logs.upload_failed(local: local_installer_log, remote: tmp_installer_log))
        next
      end

      # Copy over to log on remote
      date = DateTime.now.rfc3339
      remote_installer_log = "#{@log_dir}/installer-#{date}.log"
      remotecopylogs = Installer::Action::Execute.new(:hosts => [host], :sudo => true, :cmd => "cp #{tmp_installer_log} #{remote_installer_log}" )
      yield remotecopylogs
      unless remotecopylogs.success?
        error(hostname, @t.step.backup_logs.copy_failed(from: tmp_installer_log, to: remote_installer_log, error: remotecopylogs.error_message(hostname)))
        next
      end

      # Chmod .json
      installer_chmod_cmd = "chmod 600 #{remote_installer_log}"
      chmodlog = Installer::Action::Execute.new(:hosts => [host], :sudo => true, :cmd => installer_chmod_cmd)
      yield chmodlog
      unless chmodlog.success?
        error(hostname, @t.step.backup_logs.chmod_failed(cmd: installer_chmod_cmd, error: chmodlog.error_message(hostname)))
        next
      end

      # Copy Installer details.json
      tmp_details_json = "#{tmpdir}/details.json"
      local_details_json = "#{@local_answer_dir}/details.json"
      copydetails = Installer::Action::Upload.new(:to_host => host, :from => local_details_json, :to => tmp_details_json)
      yield copydetails
      unless copydetails.success?
        error(hostname, @t.step.backup_logs.upload_failed(local: local_details_json, remote: tmp_details_json))
        next
      end

      # Copy details.json to /etc/puppetlabs
      remote_details_json = "#{@etc_dir}/details.json"
      remotecopydetails = Installer::Action::Execute.new(:hosts => [host], :sudo => true, :cmd => "cp #{tmp_details_json} #{remote_details_json}")
      yield remotecopydetails
      unless remotecopydetails.success?
        error(hostname, @t.step.backup_logs.copy_failed(from: tmp_details_json, to: remote_details_json, error: remotecopydetails.error_message(hostname)))
        next
      end

      # Chmod details.json
      details_chmod_cmd = "chmod 600 #{@etc_dir}/details.json"
      chmoddetails = Installer::Action::Execute.new(:hosts => [host], :sudo => true, :cmd => details_chmod_cmd)
      yield chmoddetails
      unless chmoddetails.success?
        error(hostname, @t.step.backup_logs.chmod_failed(cmd: details_chmod_cmd, error: chmoddetails.error_message(hostname)))
        next
      end

      # Destroy temp log dir
      rmtmpdir = Installer::Action::Execute.new(:hosts => [host], :cmd => "rm -rf #{tmpdir}")
      yield rmtmpdir
      unless rmtmpdir.success?
        error(hostname, @t.step.backup_logs.rmdir_tmp_failed(dir: tmpdir, error: rmtmpdir.error_message(hostname)))
        next
      end
    end
  end

  def name
    :backup_logs
  end

  def description
    @t.step.backup_logs.description(hosts: @infra.hosts.values.join(', '))
  end
end
