require 'installer/action'
require 'installer/errors'

class Installer::Planner::Step::CompatibilityCheck < Installer::Planner::Step
  def initialize(infra, data={}, pre=nil, post=nil)
    raise 'No package dir provided' unless data[:packages_dir]
    @packages_dir = data[:packages_dir]
    super
  end

  def get_available_platforms
    Dir.glob("#{@packages_dir}/*").select { |f| File.directory?(f) }.map { |f| File.basename(f) }
  end

  def perform_step(&execute)
    os_data = Hash.new {|h,k| h[k] = {}}

    # Identify the Installer being used
    available_platforms = get_available_platforms

    # Defined here: https://confluence.puppetlabs.com/display/QA/Tiers+of+Supported+OS+Platforms
    supported_platforms = [
      # Tier 1 Master
      'ubuntu-14.04-amd64',
      'el-7-x86_64',
      'el-6-x86_64',
      # Tier 2 Master
      'ubuntu-12.04-amd64',
      'sles-12-x86_64',
      'sles-11-x86_64']

    @infra.hosts.each do |hostname, host|
      # Ensure PE is not already installed
      peinstalled = Installer::Action::Execute.new(:host => host, :sudo => true, :cmd => "[ -e '/opt/puppetlabs/server/pe_version' -o -x /opt/puppet/bin/puppet ]")
      yield peinstalled

      if peinstalled.success?
        peversion = Installer::Action::Execute.new(:host => host, :sudo => true, :cmd => "cat /opt/puppetlabs/server/pe_version")
        yield peversion

        if peversion.success?
          error(hostname, @t.step.compatibility_check.pe_exists.known_version(version: peversion.results[hostname][:stdout].strip))
        else
          error(hostname, @t.step.compatibility_check.pe_exists.unknown_version)
        end
      end

      # Populates host.data[:os]
      platform = Installer::Action::DetectPlatform.new(:host => host)
      yield platform

      unless platform.success?
        error(host.hostname, @t.step.compatibility_check.no_platforms_detected)
      end

      unless supported_platforms.include? host.data[:os][:tag]
        error(host.hostname, @t.step.compatibility_check.unsupported_platform(platform: host.data[:os][:tag]))
      end

      # If we haven't found the distribution or release, it's definitely not supported.
      unless host.data[:os][:distribution] && host.data[:os][:release]
        error(host.hostname, @t.step.compatibility_check.unknown_platform)
      end

      unless host.data[:os][:architecture]
        error(host.hostname, @t.step.compatibility_check.unknown_architecture)
      end
    end

    platform_tags = hosts.collect { |h| h.data[:os][:tag] if h.data[:os] }.uniq

    if platform_tags.size > 1
      error("localhost", @t.step.compatibility_check.multiple_platforms(platforms: platform_tags.join(', ')))
    elsif platform_tags.empty?
      error("localhost", @t.step.compatibility_check.no_platforms_detected)
    end

    platform_tag = platform_tags.first

    if platform_tag && !available_platforms.include?(platform_tag.to_s)
      error("localhost", @t.step.compatibility_check.platform_not_available(platform: platform_tag, available: available_platforms.join(', ')) )
    end

    os_data
  end

  def name
    :compatibility_check
  end

  def description
    @t.step.compatibility_check.description(hostname: @infra.installer_hostname, hosts: @infra.hosts.keys.join(', '))
  end
end
