require 'installer/action'
require 'installer/errors'
require 'net/ssh'

class Installer::Planner::Step::ConnectSSH < Installer::Planner::Step
  def perform_step(&stream_block)
    @infra.hosts.each do |hostname, host|
      if !host.localhost && (host.ssh_status.nil? || host.ssh_status == :error)
        if host.ssh_key_file
          begin
            verify_key_file(host.ssh_key_file, host.ssh_key_passphrase)
            info(hostname, @t.step.connect_ssh.key_file.verified(keyfile: host.ssh_key_file, hostname: hostname))
          rescue Installer::Error => e
            error(hostname,  @t.step.connect_ssh.key_file.problem(exception: e.message))
            next
          end
        end

        begin
          host.ssh_connect
          info(hostname, @t.step.connect_ssh.established(hostname: hostname))
        rescue Installer::RemoteError => e
          error(hostname, @t.step.connect_ssh.failed(exception: e.message))
        end
      end
    end
  end

  def fail_stop?
    true
  end

  def name
    "connect_ssh_#{@uuid}".to_sym
  end

  def verify_key_file(filename, passphrase=nil)
    if filename && !filename.strip.empty?
      unless File.readable?(filename)
        raise(Installer::Error, @t.step.connect_ssh.key_file.unreadable(filename: filename))
      end

      unless File.size(filename) > 0
        raise(Installer::Error, @t.step.connect_ssh.key_file.empty(filename: filename))
      end

      unless /PRIVATE KEY/.match(File.read(filename))
        raise(Installer::Error, @t.step.connect_ssh.key_file.malformatted(filename: filename))
      end

      unless (File.stat(filename).mode & 0077) == 0
        raise(Installer::Error, @t.step.connect_ssh.key_file.too_open(filename: filename))
      end

      begin
        # In lieu of keyword args, explicit variable for the argument
        ask_passphrase = false
        Net::SSH::KeyFactory.load_private_key(filename, passphrase, ask_passphrase)
      rescue OpenSSL::PKey::PKeyError, ArgumentError => e
        if /Could not parse PKey: no start line/ =~ e.message
          raise(Installer::Error, @t.step.connect_ssh.key_file.unparseable(filename: filename))
        else
          raise(Installer::Error, @t.step.connect_ssh.key_file.unknown_error(filename: filename, exception: e.message))
        end
      end
    end
  end

  def description
    @t.step.connect_ssh.description(hostname: @infra.installer_hostname, hosts: @infra.hosts.keys.join(', '))
  end
end
