require 'installer/action'
require 'installer/errors'
require 'json'

class Installer::Planner::Step::FilesystemCheck < Installer::Planner::Step
  def initialize(infra, data={}, pre=nil, post=nil)
    super
  end

  def perform_step(&execute)
    # Master needs 404M+ (call it 1G?) in /opt and /var
    # Console needs 187M+ (call it 1G?) in /opt and /var
    # PuppetDB needs 369M+ (call it 1G?) in /opt and /var
    # Postgres needs ??? (call it 10G?) /opt
    @infra.hosts.each do |hostname, host|
      # %n - File name
      # %f - Free blocks in file system
      # %S - Fundamental block size
      fsstat = Installer::Action::Execute.new(:host => host, :cmd => 'stat -f -c "{\"%n\": {\"freeblocks\": %a, \"blocksize\": %S}}" /opt /var /tmp')
      yield fsstat

      if fsstat.success?
        fs = {}
        fsstat.results[hostname][:stdout].each_line do |l|
          fs.merge!(JSON.parse(l))
        end

        fs.each do |file, stat|
          stat['freebytes'] = stat['freeblocks'] * stat['blocksize']
          stat['freekilobytes'] = stat['freebytes'] / 1024
          stat['freemegabytes'] = stat['freekilobytes'] / 1024
          stat['freegigabytes'] = stat['freemegabytes'] / 1024
        end

        min_opt_gb = 0
        min_var_gb = 0
        min_tmp_gb = 1
        roles = host.roles.map(&:name)
        if roles.include?(:master)
          min_opt_gb += 1
          min_var_gb += 1
        end

        if roles.include?(:puppetdb)
          min_opt_gb += 1
          min_var_gb += 1
        end

        if roles.include?(:console)
          min_opt_gb += 1
          min_var_gb += 1
        end

        if roles.include?(:postgres)
          min_opt_gb = 100
          min_var_gb += 1
        end

        if fs['/opt']['freegigabytes'] < min_opt_gb
          if fs['/opt']['freegigabytes'] > 0
            freespace = "#{fs['/opt']['freegigabytes']} GB"
            if roles.include?(:postgres)
              warning(hostname, @t.step.filesystem_check.freespace.database_warning(mount: '/opt', freespace: freespace, min_gb: min_opt_gb))
            else
              warning(hostname, @t.step.filesystem_check.freespace.warning(mount: '/opt', freespace: freespace, min_gb: min_opt_gb))
            end
          else
            freespace = "#{fs['/opt']['freemegabytes']} MB"
            error(hostname, @t.step.filesystem_check.freespace.unacceptable(mount: '/opt', freespace: freespace))
          end

        end

        if fs['/var']['freegigabytes'] < min_var_gb
          if fs['/var']['freegigabytes'] > 0
            freespace = "#{fs['/var']['freegigabytes']} GB"
            warning(hostname, @t.step.filesystem_check.freespace.warning(mount: '/var', freespace: freespace, min_gb: min_opt_gb))
          else
            freespace = "#{fs['/var']['freemegabytes']} MB"
            error(hostname, @t.step.filesystem_check.freespace.unacceptable(mount: '/var', freespace: freespace))
          end
        end

        if fs['/tmp']['freegigabytes'] < min_tmp_gb
          if fs['/tmp']['freegigabytes'] > 0
            freespace = "#{fs['/tmp']['freegigabytes']} GB"
            warning(hostname, @t.step.filesystem_check.freespace.warning(mount: '/tmp', freespace: freespace, min_gb: min_tmp_gb))
          else
            freespace = "#{fs['/tmp']['freemegabytes']} MB"
            error(hostname, @t.step.filesystem_check.freespace.unacceptable(mount: '/tmp', freespace: freespace))
          end
        end

      else
        error(hostname, @t.step.filesystem_check.freespace.error)
      end
    end
  end

  def name
    :filesystem_check
  end

  def description
    @t.step.filesystem_check.description(hosts: @infra.hosts.keys.join(', '))
  end
end
