require 'installer/action'
require 'installer/errors'

class Installer::Planner::Step::HardwareCheck < Installer::Planner::Step
  def initialize(infra, data={}, pre=nil, post=nil)
    super
  end

  # Requirements based off of:
  # https://docs.puppetlabs.com/pe/latest/install_system_requirements.html#hardware-requirements
  def perform_step(&stream_block)
    @infra.hosts.each do |hostname, host|
      cpucmd = "cat /proc/cpuinfo"
      cpuinfo = Installer::Action::Execute.new(:host => host, :sudo => true, :cmd => cpucmd)
      yield cpuinfo

      cores = 0
      if cpuinfo.success?
        cpuinfo.results[hostname][:stdout].each_line do |l|
          cores += 1 if /[Pp]rocessor\s+:/.match(l)
        end
      else
        warning(hostname, @t.step.hardware_check.cpu.not_found(cmd: cpucmd))
      end

      if cores == 1
        warning(hostname, @t.step.hardware_check.cpu.insufficient.one)
      elsif cores < 2
        warning(hostname, @t.step.hardware_check.cpu.insufficient.n(cores: cores))
      end

      memcmd = "cat /proc/meminfo"
      meminfo = Installer::Action::Execute.new(:host => host, :sudo => true, :cmd => memcmd)
      yield meminfo

      totalram = 0
      if meminfo.success?
        meminfo.results[hostname][:stdout].each_line do |l|
          memtotal = /MemTotal:\s+(\d+) kB/.match(l)
          totalram = memtotal[1].to_i if memtotal
        end
        totalram /= 1024
      else
        warning(hostname, @t.step.hardware_check.memory.not_found(cmd: memcmd))
      end

      if @infra.hosts.length == 1
        if totalram < 4096
          warning(hostname, @t.step.hardware_check.memory.insufficient(ram: totalram))
        end
      else
        if totalram < 16384
          warning(hostname, @t.step.hardware_check.memory.insufficient(ram: totalram))
        end
      end
    end
  end

  def name
    :hardware_check
  end

  def description
    @t.step.hardware_check.description(hosts: @infra.hosts.keys.join(', '))
  end
end
