require 'installer/action'
require 'installer/errors'

class Installer::Planner::Step::InstallPE < Installer::Planner::Step
  def initialize(infra, data={}, pre=nil, post=nil)
    super

    @host = @infra.hosts[data[:host]] if data[:host]
    @stop_on_fail = true

    raise 'Undefined remote logging directory' unless data[:log_dir]
    @log_dir = data[:log_dir]

    raise 'Undefined local installer directory' unless data[:local_installer_dir]
    @local_installer_dir = data[:local_installer_dir]

    raise 'Undefined configuration directory' unless data[:etc_dir]
    @etc_dir = data[:etc_dir]

    raise 'Undefined local answer directory' unless data[:local_answer_dir]
    @local_answer_dir = data[:local_answer_dir]
  end

  # Call from executor with a block to handle the actions.
  def perform_step(&execution_block)
    mktemp = Installer::Action::Mktemp.new(@host, "pe-installer-XXXXXXXX")
    yield mktemp
    unless mktemp.success?
      error(@host.hostname, @t.step.install_pe.create_tmp_failed(error: mktemp.error_message(@host.hostname)))
      return
    end

    installer_dir = "#{mktemp.dir}/install"
    answer_file = "#{mktemp.dir}/#{@host.hostname}.answers"

    # Ensure log dir
    mklogdir = Installer::Action::Mkdir.new(@host, @log_dir, :sudo => true)
    yield mklogdir
    unless mklogdir.success?
      error(@host.hostname, @t.step.install_pe.create_log_dir_failed(dir: @log_dir, error: mklogdir.error_message(@host.hostname)))
      return
    end

    # Ensure /etc/puppetlabs/installer dir
    mketcdir = Installer::Action::Mkdir.new(@host, @etc_dir, :sudo => true)
    yield mketcdir
    unless mketcdir.success?
      error(@host.hostname, @t.step.install_pe.create_etc_installer_failed(dir: @etc_dir, error: mketcdir.error_message(@host.hostname)))
      return
    end

    # Ensure there is an installer available to copy
    unless File.readable? "#{@local_installer_dir}/puppet-enterprise-installer"
      error("localhost", @t.step.install_pe.installer_not_found(dir: @local_installer_dir))
      return
    end

    # Copy installer files
    installer = Installer::Action::Upload.new(:to_host => @host, :from => @local_installer_dir, :to => installer_dir, :opts => {:recursive => true})
    yield installer
    unless installer.success?
      error(@host.hostname, @t.step.install_pe.installer_not_copied(host: @host.hostname))
      return
    end

    # Copy answers file and save a copy locally
    answers = @host.generate_answers
    write_local_answers(@host.hostname, answers)

    answers = Installer::Action::Upload.new(:to_host => @host, :content => answers, :to => answer_file)
    yield answers
    unless answers.success?
      error(@host.hostname, @t.step.install_pe.installer_no_answers(host: @host.hostname))
      return
    end

    # Run installer with answer file
    install = Installer::Action::Execute.new(:host => @host, :sudo => true, :cmd => "#{installer_dir}/puppet-enterprise-installer -D -a #{answer_file}")
    yield install
    unless install.success?
      error(@host.hostname, @t.step.install_pe.installer_run_failed(host: @host.hostname))
      return
    end

    # Cleanup tmp if install succeeded. If not, we want to keep these files around
    # for troubleshooting purposes.
    cleanup = Installer::Action::Execute.new(:host => @host, :sudo => true, :cmd => "rm -rf '#{mktemp.dir}'")
    yield cleanup
    unless cleanup.success?
      error(@host.hostname, @t.step.install_pe.cleanup_tmp_failed(dir: mktemp.dir, host: @host.hostname, error: cleanup.error_message(@host.hostname)))
      return
    end
  end

  def write_local_answers(hostname, answers)
    Dir.mkdir(@local_answer_dir) unless File.directory? @local_answer_dir
    File.open("#{@local_answer_dir}/#{hostname}.answers", 'w') { |f| f.puts answers }
  end

  def fail_stop?
    true
  end

  def name
    "install_pe_#{@host}"
  end

  def description
    @t.step.install_pe.description(host: @host)
  end
end
