require 'installer/action'
require 'installer/errors'

class Installer::Planner::Step::VerifyPuppet < Installer::Planner::Step
  def initialize(infra, data={}, pre=nil, post=nil)
    super

    @host = @infra.hosts[data[:host]] if data[:host]
  end

  def perform_step(&execute)
    # Get lockfile location from config
    puppetlockfile = Installer::Action::Execute.new(:host => @host, :sudo => true, :cmd => "/opt/puppetlabs/puppet/bin/puppet config print agent_catalog_run_lockfile")
    yield puppetlockfile

    if puppetlockfile.success?
      lockfile = puppetlockfile.results[@host.hostname][:stdout].strip
      puppetlockcheck = nil

      # Wait for lockfile to appear
      10.times do
        puppetlockcheck = Installer::Action::Execute.new(:host => @host, :sudo => true, :cmd => "[ -f #{lockfile} ]")
        yield puppetlockcheck
        break if puppetlockcheck.success?
        sleep 2
      end

      if puppetlockcheck && puppetlockcheck.success?
        # Wait for end of install puppet run to complete
        60.times do
          puppetlockcheck = Installer::Action::Execute.new(:host => @host, :sudo => true, :cmd => "[ -f #{lockfile} ]")
          yield puppetlockcheck
          break unless puppetlockcheck.success?
          sleep 2
        end
      end

      puppettest = Installer::Action::Execute.new(:host => @host, :sudo => true, :cmd => "/opt/puppetlabs/puppet/bin/puppet agent --test")
      yield puppettest
    end


    if !puppettest.nil? && (puppettest.success? || puppettest.results[@host.hostname][:exit_status] == 2)
      info(@host.hostname, @t.step.verify_puppet.run_success)
    else
      error(@host.hostname, @t.step.verify_puppet.run_failure)

      # Fetch all services running
      puppetservice = Installer::Action::Execute.new(:host => @host, :sudo => true, :cmd => "/opt/puppetlabs/puppet/bin/puppet resource service")
      yield puppetservice

      # Verify all services for a host are running locally
      @host.installed_services.each do |service_name|
        if puppetservice.success?
          stdout = puppetservice.results[@host.hostname][:stdout]
          service_match = /service { '(#{Regexp.quote(service_name)}(.service)?)':[^}]*ensure => '([^']+)'.*}/m.match(stdout)
          if service_match && service_match[3] == 'running'
            info(@host.hostname, @t.step.verify_puppet.service_running(service: service_name))
          elsif service_match
            error(@host.hostname, @t.step.verify_puppet.service_not_running(service: service_name, ensure: service_match[3]))
          else
            error(@host.hostname,  @t.step.verify_puppet.service_not_found(service: service_name))
          end
        else
          error(@host.hostname,  @t.step.verify_puppet.service_check_failed(service: service_name))
        end
      end

      @host.outbound.each do |h|
        rolehosts = @infra.find_hosts_with_roles([h[:role]])
        if rolehosts.any?
          rolehost = rolehosts.first.hostname
          tcpcheck = Installer::Action::Execute.new(:host => @host, :cmd => "/bin/bash -c 'echo 1 > /dev/tcp/#{rolehost}/#{h[:port]}'")
          yield tcpcheck

          if tcpcheck.success?
            info(@host.hostname, @t.step.verify_puppet.outbound_connected(host: rolehost, port: h[:port]))
          else
            error(@host.hostname, @t.step.verify_puppet.outbound_failed(host: rolehost, port: h[:port]))
          end
        else
          error(@host.hostname, @t.step.verify_puppet.outbound_no_host(role: h[:role]))
        end
      end

      @host.inbound.each do |h|
        rolehosts = @infra.find_hosts_with_roles([h[:role]])
        if rolehosts.any?
          rolehost = rolehosts.first
          tcpcheck = Installer::Action::Execute.new(:host => rolehost, :cmd => "/bin/bash -c 'echo 1 > /dev/tcp/#{@host.hostname}/#{h[:port]}'")
          yield tcpcheck

          if tcpcheck.success?
            info(@host.hostname, @t.step.verify_puppet.inbound_connected(host: rolehost, port: h[:port]))
          else
            error(@host.hostname, @t.step.verify_puppet.inbound_failed(host: rolehost, port: h[:port]))
          end
        else
          error(@host.hostname, @t.step.verify_puppet.inbound_no_host(role: h[:role]))
        end
      end

      raise Installer::RemoteError, @t.step.verify_puppet.failure_exception
    end
  end

  def name
    "verify_puppet_#{@host}"
  end

  def description
    @t.step.verify_puppet.description(host: @host)
  end
end
