;(function (global) {

if ("EventSource" in global) return;

var reTrim = /^(\s|\u00A0)+|(\s|\u00A0)+$/g;

var EventSource = function (url) {
  var eventsource = this,  
      interval = 500, // polling interval  
      lastEventId = null,
      cache = '';

  if (!url || typeof url != 'string') {
    throw new SyntaxError('Not enough arguments');
  }

  this.URL = url;
  this.readyState = this.CONNECTING;
  this._pollTimer = null;
  this._xhr = null;
  
  function pollAgain(interval) {
    eventsource._pollTimer = setTimeout(function () {
      poll.call(eventsource);
    }, interval);
  }
  
  function poll() {
    try { // force hiding of the error message... insane?
      if (eventsource.readyState == eventsource.CLOSED) return;

      // NOTE: IE7 and upwards support
      var xhr = new XMLHttpRequest();
      xhr.open('GET', eventsource.URL, true);
      xhr.setRequestHeader('Accept', 'text/event-stream');
      xhr.setRequestHeader('Cache-Control', 'no-cache');
      // we must make use of this on the server side if we're working with Android - because they don't trigger 
      // readychange until the server connection is closed
      xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');

      if (lastEventId != null) xhr.setRequestHeader('Last-Event-ID', lastEventId);
      cache = '';
    
      xhr.timeout = 50000;
      xhr.onreadystatechange = function () {
        if (this.readyState == 3 || (this.readyState == 4 && this.status == 200)) {
          // on success
          if (eventsource.readyState == eventsource.CONNECTING) {
            eventsource.readyState = eventsource.OPEN;
            eventsource.dispatchEvent('open', { type: 'open' });
          }

          var responseText = '';
          try {
            responseText = this.responseText || '';
          } catch (e) {}
        
          // process this.responseText
          var parts = responseText.substr(cache.length).split("\n"),
              eventType = 'message',
              data = [],
              i = 0,
              line = '';
            
          cache = responseText;
        
          // TODO handle 'event' (for buffer name), retry
          for (; i < parts.length; i++) {
            line = parts[i].replace(reTrim, '');
            if (line.indexOf('event') == 0) {
              eventType = line.replace(/event:?\s*/, '');
            } else if (line.indexOf('retry') == 0) {                           
              retry = parseInt(line.replace(/retry:?\s*/, ''));
              if(!isNaN(retry)) { interval = retry; }
            } else if (line.indexOf('data') == 0) {
              data.push(line.replace(/data:?\s*/, ''));
            } else if (line.indexOf('id:') == 0) {
              lastEventId = line.replace(/id:?\s*/, '');
            } else if (line.indexOf('id') == 0) { // this resets the id
              lastEventId = null;
            } else if (line == '') {
              if (data.length) {
                var event = new MessageEvent(data.join('\n'), eventsource.url, lastEventId);
                eventsource.dispatchEvent(eventType, event);
                data = [];
                eventType = 'message';
              }
            }
          }

          if (this.readyState == 4) pollAgain(interval);
          // don't need to poll again, because we're long-loading
        } else if (eventsource.readyState !== eventsource.CLOSED) {
          if (this.readyState == 4) { // and some other status
            // dispatch error
            eventsource.readyState = eventsource.CONNECTING;
            eventsource.dispatchEvent('error', { type: 'error' });
            pollAgain(interval);
          } else if (this.readyState == 0) { // likely aborted
            pollAgain(interval);
          } else {
          }
        }
      };
    
      xhr.send();
    
      setTimeout(function () {
        if (true || xhr.readyState == 3) xhr.abort();
      }, xhr.timeout);
      
      eventsource._xhr = xhr;
    
    } catch (e) { // in an attempt to silence the errors
      eventsource.dispatchEvent('error', { type: 'error', data: e.message }); // ???
    } 
  };
  
  poll(); // init now
};

EventSource.prototype = {
  close: function () {
    // closes the connection - disabling the polling
    this.readyState = this.CLOSED;
    clearInterval(this._pollTimer);
    this._xhr.abort();
  },
  CONNECTING: 0,
  OPEN: 1,
  CLOSED: 2,
  dispatchEvent: function (type, event) {
    var handlers = this['_' + type + 'Handlers'];
    if (handlers) {
      for (var i = 0; i < handlers.length; i++) {
        handlers[i].call(this, event);
      }
    }

    if (this['on' + type]) {
      this['on' + type].call(this, event);
    }
  },
  addEventListener: function (type, handler) {
    if (!this['_' + type + 'Handlers']) {
      this['_' + type + 'Handlers'] = [];
    }
    
    this['_' + type + 'Handlers'].push(handler);
  },
  removeEventListener: function (type, handler) {
    var handlers = this['_' + type + 'Handlers'];
    if (!handlers) {
      return;
    }
    for (var i = handlers.length - 1; i >= 0; --i) {
      if (handlers[i] === handler) {
        handlers.splice(i, 1);
        break;
      }
    }
  },
  onerror: null,
  onmessage: null,
  onopen: null,
  readyState: 0,
  URL: ''
};

var MessageEvent = function (data, origin, lastEventId) {
  this.data = data;
  this.origin = origin;
  this.lastEventId = lastEventId || '';
};

MessageEvent.prototype = {
  data: null,
  type: 'message',
  lastEventId: '',
  origin: ''
};

if ('module' in global) module.exports = EventSource;
global.EventSource = EventSource;
 
})(this);
// Implements an object to handle recieving and displaying streaming data from the backend
// The message recieved is in the following format:
// event: <type>
//  - phase_complete
//  - phase_error
//  - step_start
//  - step_progress
//  - step_fail
//  - step_success
//  - step_warn
// data: hash
function PhaseStream(phase) {
  this.boxID = "#stream_box";
  this.phase = phase;
  this.status = "initialized";
  this.success = true;
  this.warnings = false;

  this.completeCallbacks = $.Callbacks();
  this.es = null;

  this.handleDisplayTabs();
  window.addEventListener("hashchange", this.handleDisplayTabs, false);
}

// Method for handling the toggling of tabs between summary view
// and log view.
PhaseStream.prototype.handleDisplayTabs = function() {
  if (location.hash !== '') {
    $('a[href="' + location.hash + '"]').tab('show');
  } else {
    $('a[href="#summary"]').tab('show');
  }

  $('a[data-toggle="tab"]').on('click', function(e) {
    var $this = $(this);
    e.preventDefault();
    location.hash = $this.attr('href');
  });
};

// Add a function to be called when we complete executing a phase
PhaseStream.prototype.addCompleteCallback = function(fn) {
  this.completeCallbacks.add(fn);
};

// Creates the EventSource stream based on the phase
PhaseStream.prototype.createEventSource = function() {
  this.es = new EventSource('/execute/' + this.phase);
};

// Appends the data recieved from the event stream to the stream box
PhaseStream.prototype.appendEventData = function(data) {
  data = JSON.parse(data.data);

  if (data.action_id) {
    var preBlockElement = $("pre[data-action-id='" + data.action_id + "']");

    if (!preBlockElement.length) {
      preBlockElement = $("<pre data-action-id='" + data.action_id + "'>");
      $(this.boxID).append(preBlockElement);
    }

    data.messages.forEach( function(msg) {
      if (msg.severity != "debug")
        preBlockElement.append(msg.message);
    });
  } else {
    var time = moment.unix(data.time).format("hh:mm:ssa:");
    data.messages.forEach( function(msg) {
      if (msg.severity != "debug") {
        var template = "<span class='event'>";
        template += "<pre class='" + msg.severity + "'>" + msg.message + "</pre>";
        template = "<span class='event'>";
        template += "<pre class='" + msg.severity + "'>" + msg.message + "</pre>";
        template += "</span>";
        $(this.boxID).append($(template));
      }
    });
  }
};

// Updates the appropriate progress indicator, or creates a new one if
// appropriate.
PhaseStream.prototype.updateProgressData = function(data) {
  data = JSON.parse(data.data);
  var preBlockElement = $("pre[data-action-id='" + data.action_id + "']");

  if (preBlockElement.length) {
    var progressElement = preBlockElement.children("div[data-id='progress']");

    if (!progressElement.length) {
      progressElement = $("<div data-id='progress'>");
      preBlockElement.append(progressElement);
    }

    var pct = ("----" + data.percent).slice(-4).replace(/-/g, '&nbsp;');
    progressElement.html(pct + "%: " + data.task);
  }
};

PhaseStream.prototype.stepTransition = function(data, state) {
  data = JSON.parse(data.data);
  var step = data.name;
  var messages = data.messages;

  // The element to work with for marking status and messages
  var stepElement = $("div[data-step='" + step + "']");
  stepElement.attr('data-state', state);

  // Add any warnings or messages needed
  for (var idx in messages) {
    var msg = messages[idx];
    stepElement.append("<p data-severity='" + msg.severity + "'>[" + msg.hostname + "] " + msg.message + "</p>");
  }
};

PhaseStream.prototype.start = function() {
  this.createEventSource();

  var self = this;
  var autoscroll = function(fn) { return fn.bind(self); };

  if (window.scrollBy) {
    var pinnedToBottom = true;

    var lockToBottom = function() {
      if (pinnedToBottom) {
        window.scrollBy(0, document.body.scrollHeight);
      }
    };

    // Automatically update pinnedToBottom when the user scrolls.
    window.addEventListener('scroll', function() {
      var bottomOfDocument = document.body.scrollHeight;
      var bottomOfViewport = window.scrollY + window.innerHeight;
      var distanceToBottom = bottomOfDocument - bottomOfViewport;
      pinnedToBottom = distanceToBottom < 16;

      if (distanceToBottom < 0) {
        lockToBottom();
      }
    }, false);

    setInterval(lockToBottom, 10);
    autoscroll = function(fn) {
      return function() {
        fn.apply(self, arguments);
        lockToBottom();
      };
    };
  }

  this.es.addEventListener("step_start", autoscroll(function(m) {
    this.stepTransition(m, 'in-progress');
  }), false);

  this.es.addEventListener("step_success", autoscroll(function(m) {
    this.stepTransition(m, 'success');
  }), false);

  this.es.addEventListener("step_warn", autoscroll(function(m) {
    this.appendEventData(m);
    this.stepTransition(m, 'warn');
    this.warnings = true;
  }), false);

  this.es.addEventListener("step_error", autoscroll(function(m) {
    this.appendEventData(m);
    this.success = false;
    this.stepTransition(m, 'error');
  }), false);

  this.es.addEventListener("step_skip", autoscroll(function(m) {
    this.stepTransition(m, 'skipped');
  }), false);

  this.es.addEventListener("action_progress", autoscroll(function(m) {
    this.appendEventData(m);
  }), false);

  this.es.addEventListener("progress_indicator", autoscroll(function(m) {
    this.updateProgressData(m);
  }), false);

  this.es.addEventListener("phase_error", autoscroll(function(m) {
    this.appendEventData(m);
    this.es.close();
  }), false);

  this.es.addEventListener("phase_complete", autoscroll(function(m) {
    this.es.close();

    var status = 'error';

    if(this.success && !this.warnings) {
      status = 'success';
    } else if(this.success && this.warnings) {
      status = 'warn';
    }

    this.completeCallbacks.fire(status);
  }), false);
};
