require 'pp'
require 'json'
require 'fileutils'
require 'logger'
require 'tmpdir'
require 'pe_backup_tools/backup'
require 'pe_backup_tools/utils'
require 'pe_backup_tools/utils/conf_files'
require 'pe_backup_tools/utils/classifier'
require 'pe_backup_tools/utils/analytics_client'
require 'hocon'
require 'hocon/parser/config_document_factory'
require 'hocon/config_value_factory'
require 'hocon/config_error'

module PeBackupTools
  # This module provides the restore functionality, allowing users
  # to restore from both local and s3 backups.
  # restore: Restores puppet server from backup and configures puppet server w/ current server config
  module Restore
    def self.gpg_decrypt(backup_path)
      archive_path = File.basename(backup_path, '.gpg')
      success = system("gpg -o #{archive_path} -d #{backup_path}")
      if success && File.exist?(archive_path)
        return archive_path
      else
        raise _("Error: Failed to decrypt backup file #{backup_path}. Make sure gpg is installed and your gpg private key is imported on this host")
      end
    end

    def self.get_backup_metadata(archive_path, logfile)
      backup_metadata_string = `tar --to-stdout -xf #{archive_path} backup_metadata.json 2> #{logfile}`
      raise _('Failed to find backup metadata.') if backup_metadata_string.empty?
      JSON.parse(backup_metadata_string.strip, symbolize_names: true)
    end

    def self.unpack_tarfile(archive_path, possible_scope, scope_config, logfile, tempdir)
      included_paths_args = ''
      unless possible_scope.sort == scope_config[:scope].sort
        # If we're not using the whole scope of the backup, we only want to grab
        # the relevant files from the archive
        included_paths = scope_config[:included_paths] + scope_config[:included_dbs].map { |db| "#{db}.bin" }
        included_paths_args = included_paths.map(&:to_s).join(' ')
      end
      success = system("tar -P -C '#{tempdir}' -xvf #{archive_path} #{included_paths_args} >>#{logfile} 2>&1")
      raise _('Failed to unpack PE backup.') unless success
    end

    def self.remove_puppet_ssl_dir
      puppet_ssl_dir = '/etc/puppetlabs/puppet/ssl'
      FileUtils.remove_entry(puppet_ssl_dir) if File.directory?(puppet_ssl_dir)
    end

    def self.remove_old_certs(hostname)
      signed = "/etc/puppetlabs/puppet/ssl/ca/signed/#{hostname}.pem"
      request = "/etc/puppetlabs/puppet/ssl/ca/requests/#{hostname}.pem"
      FileUtils.rm_f(signed) if File.exist?(signed)
      FileUtils.rm_f(request) if File.exist?(request)
    end

    def self.stop_pe_services(logfile)
      %w[
        puppet
        pe-puppetdb
        pe-puppetserver
        pe-orchestration-services
        pe-console-services
      ].each do |svc|
        system("/opt/puppetlabs/bin/puppet resource service #{svc} ensure=stopped >>#{logfile} 2>&1")
      end
    end

    def self.migrate_puppet_conf(certname)
      puppet_cmd = '/opt/puppetlabs/puppet/bin/puppet'
      success = system("#{puppet_cmd} config set --section=main certname '#{certname}'")
      success = system("#{puppet_cmd} config set --section=main server '#{certname}'") if success
      success = system("#{puppet_cmd} config set --section=master certname '#{certname}'") if success
      success = system("#{puppet_cmd} config set --section=agent server_list '#{certname}:8140'") if success
      raise _('Failed to migrate puppet.conf to new certname values.') unless success
    end

    def self.migrate_user_data_conf(certname)
      migrate_enterprise_conf_file('/etc/puppetlabs/enterprise/conf.d/user_data.conf', certname)
    end

    def self.migrate_pe_conf(certname)
      migrate_enterprise_conf_file('/etc/puppetlabs/enterprise/conf.d/pe.conf', certname)
    end

    def self.migrate_enterprise_conf_file(pe_conf_path, certname)
      return if !File.exist?(pe_conf_path)

      pe_conf = Hocon::Parser::ConfigDocumentFactory.parse_file(pe_conf_path)
      new_conf_values = {
        'puppet_enterprise::profile::agent::primary_uris' => ["https://#{certname}:8140"]
      }
      pe_conf = PeBackupTools::Utils::ConfFiles.hocon_merge(pe_conf, new_conf_values)
      %w[
        puppet_enterprise::puppet_master_host
        puppet_enterprise::puppetdb_host
        puppet_enterprise::console_host
        puppet_enterprise::database_host
        puppet_enterprise::certificate_authority_host
        puppet_enterprise::pcp_broker_host
        pe_repo::master
      ].each do |setting|
        pe_conf = PeBackupTools::Utils::ConfFiles.hocon_assoc(pe_conf, setting, certname.to_s)
      end
      File.write(pe_conf_path, pe_conf.render)
    rescue Hocon::ConfigError::ConfigParseError => e
      raise _('Failed to migrate pe.conf to new certname values: %{error}.') % { error: e.message }
    end

    # MEEP uses the pe_version file to determine whether or not to run certain
    # upgrade logic but the pe_version file is managed by a PE package so we
    # don't want to overwrite it completely. The two functions below backup the
    # package version of the file before the restore, let the restore happen
    # (running any upgrade logic), and restores the package's version of the
    # file.
    def self.backup_pe_version_file(dir)
      FileUtils.cp('/opt/puppetlabs/server/pe_version', "#{dir}/pe_version")
    end

    def self.restore_pe_version_file(dir)
      FileUtils.cp("#{dir}/pe_version", '/opt/puppetlabs/server/pe_version') if File.exist?("#{dir}/pe_version")
    end

    def self.restore_database(db, logfile, tempdir)
      # Drop pglogical schema if present
      # When restoring onto a replica that has the pglogical schema already, not dropping it here will cause failures
      success = system("su - pe-postgres -s '/bin/bash' -c \"/opt/puppetlabs/server/bin/psql --tuples-only -d '#{db}' -c 'DROP SCHEMA IF EXISTS pglogical CASCADE;'\" >>#{logfile} 2>&1")
      raise _('Failed to drop pglogical schema for %{db}.') % { db: db } unless success

      tmp_db_dump = "#{tempdir}/#{db}.bin"
      success = system("su - pe-postgres -s /bin/bash -c \"/opt/puppetlabs/server/bin/psql -d '#{db}' -c 'DROP SCHEMA public CASCADE; CREATE SCHEMA public;'\" >>#{logfile} 2>&1")
      raise _('Failed to drop tables in PE database %{db}.') % { db: db } unless success
      success = system("su - pe-postgres -s /bin/bash -c \"/opt/puppetlabs/server/bin/pg_restore #{tmp_db_dump} -Fd -j4 --dbname=#{db}\" >>#{logfile} 2>&1")
      raise _('Failed to restore PE database %{db}.') % { db: db } unless success

      # Drop pglogical extension and schema (again) if present after db restore
      success = system("su - pe-postgres -s '/bin/bash' -c \"/opt/puppetlabs/server/bin/psql --tuples-only -d '#{db}' -c 'DROP SCHEMA IF EXISTS pglogical CASCADE;'\" >>#{logfile} 2>&1")
      raise _('Failed to drop pglogical schema for %{db}.') % { db: db } unless success

      success = system("su - pe-postgres -s '/bin/bash' -c \"/opt/puppetlabs/server/bin/psql --tuples-only -d '#{db}' -c 'DROP EXTENSION IF EXISTS pglogical CASCADE;'\" >>#{logfile} 2>&1")
      raise _('Failed to drop pglogical extension for %{db}.') % { db: db } unless success
    ensure
      FileUtils.rm_f(tmp_db_dump)
    end

    def self.compute_total_steps(scope, new_certname)
      steps = 13
      steps -= 1 unless scope.include?('certs')
      steps -= 1 unless scope.include?('puppetdb')
      steps -= 2 unless scope.include?('config') && new_certname
      steps -= 5 unless scope.include?('config')
      steps -= 1 unless scope.include?('puppetdb') && new_certname
      steps
    end

    def self.update_pe_classification(backup_certname, restore_certname)
      cert_data_dir = '/opt/puppetlabs/server/data/console-services/certs'
      cert_conf ||= PeBackupTools::Utils.get_cert_conf(cert_data_dir, restore_certname)
      backup_groups = PeBackupTools::Utils::Classifier.get_classification(backup_certname, restore_certname, cert_conf)['groups']
      restore_groups = PeBackupTools::Utils::Classifier.get_classification(restore_certname, restore_certname, cert_conf)['groups']
      backup_only_groups = backup_groups - restore_groups
      backup_only_groups.each do |id|
        PeBackupTools::Utils::Classifier.pin_nodes_to_group(id, [restore_certname], restore_certname, cert_conf)
      end
      PeBackupTools::Utils::Classifier.unpin_from_all([backup_certname], restore_certname, cert_conf)

      # Check for HA replica
      replica_hostnames = nil
      pe_ha_ng = PeBackupTools::Utils::Classifier.get_node_group('PE HA Master', restore_certname, cert_conf)
      if pe_ha_ng
        replica_hostnames = pe_ha_ng['classes']['puppet_enterprise::profile::database']['replica_hostnames'] - [restore_certname]
      end
      # Replace old primary with new primary and remove replica from parameter values
      PeBackupTools::Utils::Classifier.update_node_group_array_params(['PE Infrastructure Agent', 'PE Agent'],
                                                                      %w[pcp_broker_list server_list primary_uris],
                                                                      'puppet_enterprise::profile::agent',
                                                                      backup_certname,
                                                                      restore_certname,
                                                                      replica_hostnames,
                                                                      restore_certname,
                                                                      cert_conf)
      PeBackupTools::Utils::Classifier.update_node_group_array_params(['PE Master'],
                                                                      %w[puppetdb_host classifier_host classifier_client_certname],
                                                                      'puppet_enterprise::profile::master',
                                                                      backup_certname,
                                                                      restore_certname,
                                                                      replica_hostnames,
                                                                      restore_certname,
                                                                      cert_conf)
      # Remove extraneous port value from puppetdb_port and classifier_port
      pe_primary_ng = PeBackupTools::Utils::Classifier.get_node_group('PE Master', restore_certname, cert_conf)
      if pe_primary_ng
        %w[puppetdb_port classifier_port].each do |parameter|
          if pe_primary_ng['classes']['puppet_enterprise::profile::master'][parameter]
            pe_primary_ng['classes']['puppet_enterprise::profile::master'][parameter].uniq!
          end
        end
        PeBackupTools::Utils::Classifier.update_node_group(pe_primary_ng, restore_certname, cert_conf)
      end
      PeBackupTools::Utils::Classifier.set_node_group_param('PE Infrastructure', 'replicating', 'puppet_enterprise', nil, restore_certname, cert_conf)
      # returns hostname if replicas are present in backup config
      return replica_hostnames if replica_hostnames && !replica_hostnames.empty?
    end

    def self.pre_restore_interaction
      puts _('Continue with restore? Y/n')
      attempts = 0
      answer = nil
      while attempts < 3
        answer = STDIN.gets.chomp.downcase
        return if ['y', 'yes', ''].include? answer
        raise _('Restore halted by confirmation check.') if %w[n no].include? answer
        attempts += 1
      end
      raise _('Unknown response received: %{answer}.') % { answer: answer }
    end

    def self.disk_space_interaction(space_needed, free_space, mount_point)
      if free_space && free_space > space_needed
        puts _('Disk space available.')
        return
      end
      humanized_space_needed = PeBackupTools::Utils.humanize_size(space_needed)
      if free_space
        puts _('There is not enough space to restore.') + ' ' + _('Estimated restore size is %{space_needed}, and %{mount_point} has %{free_space} of space available.') % { space_needed: humanized_space_needed, mount_point: mount_point, free_space: PeBackupTools::Utils.humanize_size(free_space) } + ' ' + _('Use the --force option to proceed without a disk space check.')
        puts _('Would you like to proceed with estimated insufficient space? Y/n')
      else
        puts _('Unable to determine if there is enough space to restore.') + '  ' + _('Estimated restore size is %{space_needed}.') % { space_needed: humanized_space_needed } + ' ' + _('Use the --force option to proceed without a disk space check.')
        puts _('Would you like to proceed without checking the available disk space? Y/n')
      end
      attempts = 0
      answer = nil
      while attempts < 3
        answer = STDIN.gets.chomp.downcase
        return if ['y', 'yes', ''].include? answer
        raise _('Restore halted by disk space check.') if %w[n no].include? answer
        attempts += 1
      end
      raise _('Unknown response received: %{answer}.') % { answer: answer }
    end

    def self.estimate_restore_size(backup_size, scopes)
      scopes.inject(0) { |sum, scope| sum + backup_size[scope.to_sym][:total_size] }
    end

    def self.get_tempdir(options)
      File.expand_path(options[:tempdir] || ENV['TMPDIR'] || '/tmp')
    end

    def self.restore(options)
      timestamp = Time.now.utc.strftime('pe_restore-%Y-%m-%d_%H.%M.%S_UTC')
      dir = Dir.mktmpdir
      archive_path = options[:backup]
      tempdir = get_tempdir(options)
      logdir = File.expand_path(options[:logdir] || '/var/log/puppetlabs/pe-backup-tools')
      logfilename = timestamp + '.log'
      logger, logfile = PeBackupTools::Utils.configure_logging(logdir, logfilename)

      raise _('Specified local backup file does not exist.') unless File.readable?(archive_path)

      if File.extname(archive_path) == '.gpg'
        decrypted_file = gpg_decrypt(archive_path)
        archive_path = decrypted_file
      end

      logger.info(_('Restoring backup file %{archive_path}') % { archive_path: archive_path })

      logger.info(_('Retrieving metadata from backup file'))
      backup_metadata = get_backup_metadata(archive_path, logfile)
      backup_scope = backup_metadata[:scope_config][:scope]
      if options[:scope] == 'all'
        options[:scope] = backup_scope
      else
        invalid = options[:scope].reject { |scope| backup_scope.include?(scope) }
        unless invalid.empty?
          error = [n_('Backup only contains scope: %{valid}', 'Backup only contains scopes: %{valid}.', backup_scope.length) % { valid: backup_scope.join(', ').chomp },
                   n_('Invalid scope option: %{invalid}.', 'Invalid scope options: %{invalid}.', invalid.length) % { invalid: invalid.join(', ').chomp }].join("\n")
          raise error
        end
      end

      scope_config = PeBackupTools::Backup.generate_scope_config(options)
      timings = {}
      restore_certname = PeBackupTools::Utils.get_certname
      backup_certname = backup_metadata[:certname]
      step_printer = PeBackupTools::Utils::StepPrinter.new(compute_total_steps(scope_config[:scope],
                                                                               backup_certname != restore_certname))

      backup_pe_version_file(dir)

      space_needed = estimate_restore_size(backup_metadata[:uncompressed_size], scope_config[:scope])
      humanized_space_needed = PeBackupTools::Utils.humanize_size(space_needed)
      unless options[:force]
        timings['disk_space_check'] = PeBackupTools::Utils.benchmark do
          puts _('Checking for disk space to restore.')
          free_space, mount_point = PeBackupTools::Utils.extract_available_space_and_mount_point(dir)
          disk_space_interaction(space_needed, free_space, mount_point)
        end
      end

      pre_restore_summary = ['',
                             _('Restoring backup file %{path}') % { path: archive_path },
                             '  ' + _('Scope: %{scope}') % { scope: scope_config[:scope].join(', ').chomp },
                             '  ' + _('Backup size: %{space_needed}') % { space_needed: humanized_space_needed },
                             '  ' + _('Backup PE version: %{pe_version}') % { pe_version: backup_metadata[:pe_version] },
                             '',
                             _('Restoring to:'),
                             '  ' + _('Primary: %{certname}') % { certname: restore_certname },
                             '  ' + _('PE version: %{pe_version}') % { pe_version: PeBackupTools::Utils.get_pe_version },
                             '',
                             _('Temp dir set to:'),
                             '  ' + tempdir.to_s,
                             _('This typically takes about 20 minutes, but in some cases could take several hours.'),
                             '',
                             ''].join("\n")
      puts pre_restore_summary
      pre_restore_interaction unless options[:force]

      puts ''
      puts _('Log messages will be saved to %{logfile}') % { logfile: logfile }
      puts ''

      # Make our temp directory if non-existent
      FileUtils.mkdir_p(tempdir) unless Dir.exist?(tempdir)

      timings['stop_services'] = PeBackupTools::Utils.benchmark do
        step_printer.step(_('Stopping PE related services'))
        stop_pe_services(logfile)
      end

      if scope_config[:scope].include?('certs')
        # The previous PE install has agent certificates hanging around and we
        # need to remove them if we're restoring certs
        step_printer.step(_('Cleaning the agent certificates from previous PE install'))
        remove_puppet_ssl_dir
      end

      pe_databases = scope_config[:included_dbs]
      begin
        timings['file_system_restore'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Restoring PE file system components'))
          unpack_tarfile(archive_path, backup_scope, scope_config, logfile, tempdir)
        end
        pe_databases.each do |db|
          timings["#{db.tr('-', '_')}_restore"] = PeBackupTools::Utils.benchmark do
            step_printer.step(_('Restoring the %{db} database') % { db: db })
            restore_database(db, logfile, tempdir)
          end
        end
      ensure
        # This is a fail safe in case we fail to run restore_database on
        # all the databases which should clean up each binary individually
        tmp_db_dumps = pe_databases.map { |db| "#{tempdir}/#{db}.bin" }
        FileUtils.rm_f(tmp_db_dumps)
      end

      if scope_config[:scope].include?('config') && restore_certname != backup_certname
        timings['config_migration'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Migrating PE configuration for new server'))
          migrate_puppet_conf(restore_certname)
          migrate_pe_conf(restore_certname)
          migrate_user_data_conf(restore_certname)
        end

        if scope_config[:scope].include?('certs')
          remove_old_certs(restore_certname)
        end
      end

      timings['puppet_infrastructure_configure'] = PeBackupTools::Utils.benchmark do
        step_printer.step(_('Configuring PE on newly restored primary'))
        environment = options[:pe_environment] || Puppet[:environment]
        PeBackupTools::Utils.puppet_infrastructure_configure(environment, logfile)
      end
      r10k_remote = nil
      preexisting_replica = nil
      if scope_config[:scope].include?('config') && restore_certname != backup_certname
        timings['classification_updates'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Updating PE Classification for new server'))
          preexisting_replica = update_pe_classification(backup_certname, restore_certname)
          r10k_remote = `puppet lookup puppet_enterprise::profile::master::r10k_remote`
          migrate_puppet_conf(restore_certname)
        end
      end

      if scope_config[:scope].include?('puppetdb') && restore_certname != backup_certname
        timings['deactivate_old_master'] = PeBackupTools::Utils.benchmark do
          step_printer.step(_('Deactivating the old primary node'))
          PeBackupTools::Utils.puppet_node_deactivate(backup_certname, logfile)
        end
      end

      timings = timings.merge(total: PeBackupTools::Utils.get_total_runtime(timings))

      cert_conf = PeBackupTools::Utils.get_agent_cert_conf(restore_certname)
      analytics_url = "https://#{restore_certname}:8140/analytics/v1"
      PeBackupTools::Utils::AnalyticsClient.store_event(analytics_url, cert_conf, 'puppet-backup.restore',
                                                        scope: scope_config[:scope],
                                                        force: options[:force] ? true : false,
                                                        timings: timings)

      logger.info(_('Restore succeeded (size: %{space_needed}, scope: %{scope}, time: %{timings})') % { space_needed: humanized_space_needed, scope: scope_config[:scope], timings: PeBackupTools::Utils.humanize_time(timings[:total]) })
      {
        command: 'restore',
        status: 'success',
        scope: scope_config[:scope],
        size: space_needed,
        runtime: timings,
        preexisting_replica: preexisting_replica,
        r10k_remote: r10k_remote
      }
    rescue RuntimeError => e
      logger.error(_('Restore failed with: %{error}') % { error: e.message }) if logger
      {
        command: 'restore',
        status: 'failed',
        error_description: e.message,
        logfile: logfile
      }
    ensure
      restore_pe_version_file(dir)
      FileUtils.remove_entry(dir)
    end
  end
end
