#! frozen_string_literal: true

require 'bolt/error'

# This is a copy of https://github.com/puppetlabs/bolt/blob/1.48.0/bolt-modules/boltlib/lib/puppet/functions/set_feature.rb
#
# The original function:
#
# Sets a particular feature to present on a target.
#
# Features are used to determine what implementation of a task should be run.
# Currently supported features are
# - powershell
# - shell
# - puppet-agent
#
# ---
#
# This copy of the function is the same if Bolt is evaluating the plan, but
# since the Orchestrator raises a ValidationError, since it doesn't implement
# set_feature(), it silently swallows that, allowing the plan to proceed.
#
# Our use of set_feature was to ensure that facter and service tasks ran the
# puppet-agent implementations (since all the PE nodes have the agent, but we
# don't have an inventory file declaring that).
#
# This is irrelevant when running with the Orchestrator. In that case
# puppet-agent is guaranteed since the agent is assumed to be installed for the
# node to have pxp-agent and be in Orchestrator's inventory.
Puppet::Functions.create_function('enterprise_tasks::set_feature') do
  # @param target The Target object to add features to. See {get_targets}.
  # @param feature The string identifying the feature.
  # @param value Whether the feature is supported.
  # @return The target with the updated feature
  # @example Add the puppet-agent feature to a target
  #   set_feature($target, 'puppet-agent', true)
  dispatch :set_feature do
    param 'Target', :target
    param 'String', :feature
    optional_param 'Boolean', :value
    return_type 'Target'
  end

  def set_feature(target, feature, value = true)
    unless Puppet[:tasks]
      raise Puppet::ParseErrorWithIssue
        .from_issue_and_stack(Bolt::PAL::Issues::PLAN_OPERATION_NOT_SUPPORTED_WHEN_COMPILING, action: 'set_feature')
    end

    inventory = Puppet.lookup(:bolt_inventory)
    executor = Puppet.lookup(:bolt_executor)
    executor.report_function_call(self.class.name)

    begin
      inventory.set_feature(target, feature, value)
    rescue Bolt::Inventory::ValidationError # rubocop:disable Lint/HandleExceptions
      # Orchestrator will throw this, just ignore and continue.
    end

    target
  end
end
