#!/opt/puppetlabs/puppet/bin/ruby
# frozen_string_literal: true

require_relative '../files/enterprise_task_helper.rb'
require 'json'
require 'fileutils'

class RemoveOldPackages < EnterpriseTaskHelper

  def remove_old_packages(pe_version, agent_version)
    packages_dir = '/opt/puppetlabs/server/data/packages/public'
    staging_dir = '/opt/puppetlabs/server/data/staging'
    removed_pe_dirs = []
    removed_agent_dirs = []
    # Guard against deleting current version. On database node the 'current' symlink doesn't exist
    if Dir.exist?(packages_dir)
      if File.exist?("#{packages_dir}/current") && File.symlink?("#{packages_dir}/current")
        current_ver = File.basename(File.readlink("#{packages_dir}/current"))
      else
        current_ver = pe_version
      end
      version = Gem::Version.new(pe_version)
      Dir.chdir(packages_dir) do
        Dir.glob('*').select { |f| f.match(/\A\d+\.\d+\.\d+/) }.each do |dir|
          if File.directory?(dir) && dir != current_ver && (Gem::Version.new(dir) < version)
            FileUtils.rm_rf(dir)
            removed_pe_dirs << "#{dir}"
          end
        end
      end
    end

    # Agent version is grabbed from the primary, and we clean up old agent packages
    #   left over in the staging dir. The current agent package is managed by puppet
    #   in this dir, but this task does not clean it up regardless. So one package
    #   will remain present in the directory.
    if Dir.exist?(staging_dir)
      Dir.chdir(staging_dir) do
        Dir.glob('*').select { |f| f.match(/pe_repo-puppet-agent-\d+\.\d+\.\d+/) }.each do |dir|
          # The directories are of the form pe_repo-puppet-agent-<agent_ver>
          #   so this transformation just grabs the agent version from the dir
          #   to compare against the version passed in to this task
          dir_version_str = dir.sub('pe_repo-puppet-agent-', '')
          dir_version = Gem::Version.new(dir_version_str) if Gem::Version.correct?(dir_version_str)

          # If dir_version is a valid version and is older than the current puppet-agent version, remove it
          if File.directory?(dir) && dir_version && (dir_version < Gem::Version.new(agent_version))
            FileUtils.rm_rf(dir)
            removed_agent_dirs << "#{dir_version}"
          end
        end
      end
    end

    return removed_pe_dirs.join(', '), removed_agent_dirs.join(', ')
  end

  def task(pe_version:, agent_version:, **_kwargs)
    removed_pe_versions, removed_agent_versions = remove_old_packages(pe_version, agent_version)
    result = {
      success: true,
      removed_pe_versions: removed_pe_versions,
      removed_agent_versions: removed_agent_versions,
    }
    result.to_json
  end
end

RemoveOldPackages.run if __FILE__ == $PROGRAM_NAME
