module Puppet::Parser::Functions
  newfunction(:get_hiera_overrides, :type => :rvalue, :arity => 2, :doc => <<-EOS
    Given a list of parameters and a base module namespace, will return a hash of
    parameters that a user has specified for that node in hiera.

    This is useful for creating a hash of parameters to use with `pe_node_groups`
    for ongoing classification.

    Example:

    $params_i_care_about = [
      'foo',
      'bar',
    ]

    $user_classified = get_hiera_overrides('puppet_enterprise', $params_i_care_about)

    pe_node_group { 'PE Infrastructure':
      parent  => 'All Nodes',
      classes => {
        'puppet_enterprise' => $user_classified,
      }
    }

  EOS
  ) do |args|

    namespace = args[0]
    params = args[1]

    if !namespace.kind_of?(String)
      raise Puppet::ParseError, ("get_hiera_overrides(): Namespace Argument must be of type String")
    end

    if !params.kind_of?(Array)
      raise Puppet::ParseError, ("get_hiera_overrides(): Parameters Argument must be of type Array")
    end

    hiera_overrides = {}

    params.each do |param|
      value = call_function('lookup', ["#{namespace}::#{param}", { 'default_value' => nil }])
      hiera_overrides[param] = value if !value.nil?
    end

    hiera_overrides
  end
end
