require 'puppet/indirector/face'
require 'puppet/application/apply'
require 'puppet/util/command_line'
require 'puppet/util/pe_node_groups'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/util/apply'
require 'puppet_x/util/infrastructure_error'
require 'puppet_x/util/service_status'
require 'puppet_x/util/classification'
require 'puppet_x/util/classifier'
require 'puppet_x/util/ca'
require 'puppet_x/util/code_manager'
require 'puppet_x/util/interview'
require 'puppet_x/util/stringformatter'
require 'fileutils'

Puppet::Face.define(:infrastructure, '1.0.0') do
  extend PuppetX::Puppetlabs::Meep::Util

  action :promote do
    summary _('Permanently promotes a replica node to be the new primary.')
    description <<-EOT
      Promote a PE node from a replica to a primary.
    EOT

    arguments "<role>"

    option('-y', '--yes') do
      summary _('Answer yes to all confirmations.')
      default_to {nil}
    end

    option('--print-config') do
      summary _("Print the current configuration")
      description <<-EOT
        Shows the current configuration of HA related settings
      EOT
    end

    option('--topology TOPOLOGY') do
      summary _("The topology of your PE installation")
      description <<-EOT
      The topology of your PE installation:
        mono - All PE services run on the primary and all agents connect to that primary.
        mono-with-compile - All PE services run on the primary but agents connect to compilers.
      EOT
    end

    option('--skip-agent-config') do
      summary _("Do not change agent settings")
      default_to {nil}
      description PuppetX::Util::String::Formatter.indent(
        _("Do not change agent settings when enabling replica. Agent failover settings must be configured by the user."))
    end

    option('--pcp-brokers URIS') do
      summary _("Hostnames and ports of your PCP Brokers")
      description <<-EOT
        hostname:port,hostname:port
      EOT
    end

    option('--infra-pcp-brokers URIS') do
      summary _("Hostnames and ports of your PCP Brokers for PE infrastructure")
      description <<-EOT
        hostname:port,hostname:port
      EOT
    end

    option('--puppetdb-termini URIS') do
      summary _("Hostnames and ports of your PuppetDBs")
      description <<-EOT
        hostname:port,hostname:port
      EOT
    end

    option('--classifier-termini URIS') do
      summary _("Hostnames and ports of your Node Classifiers")
      description <<-EOT
        hostname:port,hostname:port
      EOT
    end

    option('--agent-server-urls URIS') do
      summary _("Hostnames and ports of your Primary Servers to run your Puppet Agents against")
      description <<-EOT
        hostname:port,hostname:port
      EOT
    end

    option('--infra-agent-server-urls URIS') do
      summary _("Hostnames and ports your Primary Servers to run your Puppet Infrastructure Agents against")
      description <<-EOT
        hostname:port,hostname:port
      EOT
    end

    option('--primary-uris URIS') do
      summary _("Hostnames and ports of your Primary Servers to connect PXP agents to")
      description <<-EOT
        hostname:port,hostname:port
      EOT
    end

    option('--infra-primary-uris URIS') do
      summary _("Hostnames and ports your Primary Servers to connect your Puppet Infrastructure PXP agents to")
      description <<-EOT
        hostname:port,hostname:port
      EOT
    end

    when_invoked do |role, options|
      if role != 'replica'
        raise PuppetX::Util::InfrastructureUnknownRoleError.new(role, ["replica"]) # rubocop:disable GetText/DecorateFunctionMessage
      end

      # We aren't in enterprise_tasks here, but this will allow us to not clobber another
      # PE infra management plan run
      enterprise_tasks_agent_lockfile = '/opt/puppetlabs/enterprise_tasks_agent_run.lock'
      replica_certname = Puppet[:certname]
      ssl_dir = '/etc/puppetlabs/puppet/ssl'
      old_ca_dir = "#{ssl_dir}/ca"
      new_ca_dir = '/etc/puppetlabs/puppetserver/ca'

      config = PuppetX::Util::ServiceStatus.load_services_config()
      required_services = PuppetX::Util::ServiceStatus.replica_services()
      services = PuppetX::Util::ServiceStatus.validate_and_select_services_from_config(config, 'promote', replica_certname, required_services)

      nc_service = services[:classifier]
      Puppet.debug(_("Contacting the classifier service on %{url}...") % { url: "https://#{nc_service[:server]}:#{nc_service[:port].to_i}/#{nc_service[:prefix]}" })
      nc = Puppet::Util::Pe_node_groups.new(nc_service[:server], nc_service[:port].to_i, "/#{nc_service[:prefix]}")
      all_groups = nc.get_groups()
      master_certname = PuppetX::Util::Classification.find_primary_master(all_groups)

      if options[:print_config]
        puts _('Current configuration:')
        puts PuppetX::Util::Classification.print_config(all_groups)
        return
      end

      if PuppetX::Util::Interview.any_custom_flags?(options)
        PuppetX::Util::Interview.validate_infra_flags('promote', options)
        infra_conf = PuppetX::Util::Interview.parse_infra_flags(options)
      else
        conf = PuppetX::Util::Classification.get_config(all_groups)
        infra_conf = PuppetX::Util::Interview.promote_interview(conf, master_certname, replica_certname, options)
      end

      puts _("Promoting %{certname} as a primary...") % { certname: replica_certname }

      with_agent_disabled('puppet infrastructure promote replica in progress') do
        puts _("Copying backup Certificate Authority data...")
        # The broken symlink that's present in primary-ssl-data when the primary has
        # a migrated CA dir will trip up FileUtils.cp_r, so we delete it and recreate
        # it once the CA dir is in place.
        ssl_data_ca = '/opt/puppetlabs/server/data/primary-ssl-data/ca'
        is_ca_migrated = File.symlink?(ssl_data_ca)
        FileUtils.rm_f(ssl_data_ca) if is_ca_migrated

        # :preserve => false is intentional, we want to keep the current perms
        # of any files or directories in /etc/puppetlabs/puppet/ssl
        # This is due to EZ-83, some work is tracked in PE-18028
        FileUtils.cp_r(Dir.glob('/opt/puppetlabs/server/data/primary-ssl-data/*'),
                      ssl_dir,
                      :preserve => false)
        FileUtils.chown_R('pe-puppet', 'pe-puppet', ssl_dir)

        if is_ca_migrated
          FileUtils.mkdir_p(new_ca_dir)
          FileUtils.cp_r(Dir.glob('/opt/puppetlabs/server/data/primary-ca-data/*'),
                      new_ca_dir,
                      :preserve => false)
          FileUtils.chown_R('pe-puppet', 'pe-puppet', new_ca_dir)
          FileUtils.ln_s(new_ca_dir, old_ca_dir)
        end

        puts _("Copying code manager data...")
        # NOTE: This works because puppet infrastructure does not override the codedir,
        # even though it does override the environmentpath.
        PuppetX::Util::CodeManager.copy_client_data_to_storage_service(Puppet[:codedir])

        puts _("Advancing CA serial number by 1000...")
        # We can still use old_ca_dir here since a migrated CA dir has a symlink in the old location.
        PuppetX::Util::CA.advance_ca_serial("#{old_ca_dir}/serial", 1000)

        pe_infrastructure_group = PuppetX::Util::Classifier.find_group(all_groups, 'PE Infrastructure')
        puppet_enterprise = pe_infrastructure_group['classes']['puppet_enterprise']

        puts _("Applying database configuration to make classification writable...")
        # The notify of pe-puppetserver here isn't needed for this particular Puppet apply,
        # but is needed for the following Puppet agent run. This is because if we moved the CA files
        # into a location that is not the default for Puppet's cadir setting, puppetserver will
        # not pick up the updated cadir setting in order to set up the primary-ca-data file sync's
        # staging_dir correctly, so we need to restart the service. 
        manifest = <<-EOS
        class { 'puppet_enterprise':
          puppet_master_host => '#{replica_certname}',
        }
        service { 'pe-puppetserver': }
        include puppet_enterprise::params
        class { 'pe_manager::promote':
          classifier_database_write_user => #{PuppetX::Util::Apply.quoted_or_undef(puppet_enterprise['classifier_database_write_user'])},
          rbac_database_write_user       => #{PuppetX::Util::Apply.quoted_or_undef(puppet_enterprise['rbac_database_write_user'])},
          activity_database_write_user   => #{PuppetX::Util::Apply.quoted_or_undef(puppet_enterprise['activity_database_write_user'])},
          classifier_database_name       => #{PuppetX::Util::Apply.quoted_or_undef(puppet_enterprise['classifier_database_name'])},
          rbac_database_name             => #{PuppetX::Util::Apply.quoted_or_undef(puppet_enterprise['rbac_database_name'])},
          activity_database_name         => #{PuppetX::Util::Apply.quoted_or_undef(puppet_enterprise['activity_database_name'])},
          orchestrator_database_name     => #{PuppetX::Util::Apply.quoted_or_undef(puppet_enterprise['orchestrator_database_name'])},
          inventory_database_name        => #{PuppetX::Util::Apply.quoted_or_undef(puppet_enterprise['inventory_database_name'])},
          notify                         => Service['pe-puppetserver'],
        }
        EOS
        Puppet.debug(_("Applying the following Puppet manifest..."))
        Puppet.debug(manifest)
        PuppetX::Util::Apply.run_puppet_apply(manifest)


        puts _("Updating classification of %{replica_certname} to be a Primary...") % { replica_certname: replica_certname }
        primary_master_group_ids = PuppetX::Util::Classification.find_primary_master_group_ids(nc, all_groups, master_certname)
        Puppet.debug(_("Found the following PE classification ids for the primary: %{ids}") % { ids: primary_master_group_ids })

        configure_agent = !options[:skip_agent_config]
        PuppetX::Util::Classification.promote_replica(nc, all_groups, replica_certname, primary_master_group_ids, infra_conf, master_certname, configure_agent)
        # In order to ensure that promotion is idempotent, we don't demote the
        # replica if the replica was already classified as the primary master from
        # a previous incantation of the promote CLI
        unless master_certname == replica_certname
          puts _("Updating classification to demote %{certname} as a primary...") % { certname: master_certname }
          primary_master_group_ids.each do |group_id|
            nc.unpin_nodes_from_group(group_id, [master_certname])
          end
          puts _("Successfully removed %{certname} as a primary.") % { certname: master_certname }
        end

        puts _("Updated classification")

        puts _("Running the Puppet agent to install and configure %{certname} as a primary...") % { certname: replica_certname }
        PuppetX::Util::Shell.run_and_echo_out("/opt/puppetlabs/bin/puppet agent -t --server_list #{replica_certname} --detailed-exitcodes --agent_disabled_lockfile #{enterprise_tasks_agent_lockfile}")
        agent_run_exitstatus = $?.exitstatus

        puts _("Deactivating %{certname} in PuppetDB...") % { certname: master_certname }

        Puppet::Face[:node, :current].purge(master_certname)

        # Regenerate console certificate and SAML certificate so that the SAN matches the new promoted replica's hostname
        # See PE-27485 for more details; this is for OSX Catalina cert guideline compliance
        cert_artifact_dirs = ['certs', 'public_keys', 'private_keys']
        cert_artifact_dirs.each do |dir|
          FileUtils.rm_rf("#{ssl_dir}/#{dir}/console-cert.pem")
          FileUtils.rm_rf("#{ssl_dir}/#{dir}/saml-cert.pem")
        end
        PuppetX::Util::Shell.run_and_echo_out('/opt/puppetlabs/bin/puppetserver ca clean --certname console-cert')
        PuppetX::Util::Shell.run_and_echo_out('/opt/puppetlabs/bin/puppetserver ca clean --certname saml-cert')
        FileUtils.rm_rf("#{puppet_enterprise['console_services_ssl_dir']}/console-cert.*")
        FileUtils.rm_rf("#{puppet_enterprise['console_services_ssl_dir']}/saml-cert.*")

        if agent_run_exitstatus == 0 or agent_run_exitstatus == 2
          # There are exported resources that only show up in PuppetDB after the
          # first puppet run so we run puppet a second time to ensure that those
          # resources make it to the new replicas catalog
          puts _("Running the Puppet agent to ensure that %{certname} was configured properly...") % { certname: replica_certname }
          PuppetX::Util::Shell.run_and_echo_out("/opt/puppetlabs/bin/puppet agent -t --detailed-exitcodes --agent_disabled_lockfile #{enterprise_tasks_agent_lockfile}")

          if configure_agent
            puts PuppetX::Util::String::Formatter.join_and_wrap_no_indent([
              "\n",
              _("Agent configuration will be updated the next time each agent runs to enable replica failover."),
              "\n",
              _("If you wish to immediately run Puppet on all your agents, you can do so with this command:"),
              "\n",
              "puppet job run --no-enforce-environment --query 'nodes {deactivated is null and expired is null}'"])
          end

          second_agent_run_exitstatus = $?.exitstatus
          if second_agent_run_exitstatus == 0 or second_agent_run_exitstatus == 2
            puts _("Running recover_configuration on %{replica_certname} to ensure updating of user_data.conf to match configuration on original primary...") % {replica_certname: replica_certname}
            PuppetX::Util::Shell.run_and_echo_out('/opt/puppetlabs/bin/puppet-infrastructure recover_configuration')
            recover_config_exitstatus = $?.exitstatus
            if recover_config_exitstatus == 0 or recover_config_exitstatus == 2
              return
            else
              exit(recover_config_exitstatus)
            end
          else
            exit(second_agent_run_exitstatus)
          end
        else
          exit(agent_run_exitstatus)
        end
      end
    end
  end
end
