# coding: utf-8
require 'hiera'
require 'puppet/util/pe_node_groups'
require 'puppet/util/pe_conf/recover/puppetdb'
require 'puppet_x/puppetlabs/meep/hiera_adapter'
require 'puppet/node/facts'

class Puppet::Util::Pe_conf
  # Module that provides some helper methods for use in recovering a
  # customer's classification overrides from the PE Node Classifier,
  # Puppetdb and Hiera into a set of pe.conf files for use during a
  # `puppet infrastructure configure` run.
  class Recover

    # Returns a hash containing a map of `puppet_enterprise` module parameters
    # to certnames by querying puppetdb for all certnames that have a certain
    # profile applied to it.
    #
    # Future versions of this may wish to also fall back to the NC and grab
    # values from the `PE Infrastructure` group. Currently PuppetDB is the
    # "safer" choice as a handful of customers are known to not use the NC.
    #
    # @return [Hash] A hash of ALL puppet_enterprise parameters and their values
    def pe_conf_from_puppetdb
      Puppet.debug(_('Creating base pe.conf from puppetdb'))

      # In a split install or LEI environment, this will return multiple
      # resources, for the first version, just grab the first instance as
      # the values on `puppet_enterprise` shouldn't differ between core infra nodes.
      #
      # Future work should be done here to handle iterating over all returned nodes
      # and dealing with any discrepancies in results.
      resources = Puppet::Util::Pe_conf::Recover::Puppetdb.get_base_pe_class

      if !resources.empty?
        base_params = resources.first['parameters']
      else
        base_params = {}
      end

      base_params.inject({}) do |pe_conf, entry|
        param, value = entry
        pe_conf["puppet_enterprise::#{param}"] = value
        pe_conf
      end
    end

    # Retrieves the environment that the given node lives in.
    # This data is retrieved from the node terminus, using an ENC
    # if it is set.
    #
    # @return [String] Environment for the node 
    def get_node_environment(certname)
      # We need to pass in facts to get_node or else it will use Facter to look them up.
      facts = Puppet::Node::Facts.new(certname, facts_for_node(certname))
      node = PuppetX::Puppetlabs::Meep::HieraAdapter.get_node(certname, Puppet.lookup(:current_environment), get_node_terminus, facts, get_external_nodes)
      node.environment_name.to_s
    end

    # Retrieves a list of core infra nodes from puppetdb
    #
    # A node is considered 'core infra' if it is applied with one of the following
    # profiles:
    #
    # Certificate Authority
    # Master
    # Console
    # PuppetDB
    #
    # @return [Array] A list of certnames
    def get_core_infra_nodes
      results = Puppet::Util::Pe_conf::Recover::Puppetdb.get_pe_infra_nodes
      results.map { |r| r['certname'] }.uniq
    end

    # Retrieves a list of infra nodes with a given class from puppetdb

    def get_infra_nodes_with_class(classname)
      results = Puppet::Util::Pe_conf::Recover::Puppetdb.get_pe_infra_nodes_with_class(classname)
      results.map { |r| r['certname'] }.uniq
    end

    # Takes a hash of parameters and returns back a hash that only
    # contains 'required' `puppet_enterprise` parameters, such as hostnames.
    def extract_required_pe_conf_params(pe_conf)
      required_params = [
        'puppet_enterprise::puppet_master_host',
        'puppet_enterprise::certificate_authority_host',
        'puppet_enterprise::pcp_broker_host',
        'puppet_enterprise::console_host',
        'puppet_enterprise::puppetdb_host',
        'puppet_enterprise::database_host',
      ]

      pe_conf.select do |k,v|
        required_params.include?(k)
      end
    end

    # Iterates over the specified parameters to find any that the user has set
    # in hiera.
    #
    # @param node_name   [String]  The name of the node to find overrides for
    # @param params      [Array]  A list of fully namespaced parameters
    # @param facts       [Hash]   A hash of facts for use in creating the hiera scope
    # @param environment [String] The puppet environment to perform the lookup in
    # @param node_terminus [String] The currently configured node_terminus
    # @param external_nodes_setting [String] The currently configured external_nodes_setting
    #
    # @return [Hash] A hash containing any parameters and values found, or an empty hash
    #                if no overrides where found.
    def find_hiera_overrides(node_name, params, facts, environment, node_terminus, external_nodes_setting = nil)
      # Because we are manually constructing a node here, we need to manually
      # set the facts to whatever is in puppetdb, and we also need to manually
      # set the clientcert parameter to ensure that the trusted facts are
      # populated as expected for lookups.
      node_facts = Puppet::Node::Facts.new(node_name, facts.dup)
      Puppet.override(trusted_information: node_facts.values['trusted']) do
        current_node = PuppetX::Puppetlabs::Meep::HieraAdapter.get_node(node_name, environment, node_terminus, node_facts, external_nodes_setting)
        current_node.parameters["clientcert"] = node_name

        scope = PuppetX::Puppetlabs::Meep::HieraAdapter.generate_scope(current_node)
        Puppet.override(:global_scope => scope) do
          return params.each_with_object({}) do |param, out_hash|
            v = hiera_adapter.lookup(param, scope)
            next if v.nil?
            out_hash[param] = v
          end
        end
      end
    end

    # Queries PuppetDB for the last submitted facts for a node.
    #
    # @param certname    [String] The certificate name of the node you wish to
    #                             retrieve the facts for.
    #
    # @return [Hash] A hash of any facts found, or an empty hash.
    def facts_for_node(node)
      Puppet.debug(_("Retrieving facts for %{node} from PuppetDB") % { node: node })
      facts = Puppet::Util::Pe_conf::Recover::Puppetdb.get_facts_for_node(node)
      facts.inject({}) do |facts_hash, entry|
        name = entry["name"]
        value = entry["value"]
        facts_hash[name] = value
        facts_hash
      end
    end

    # Queries PuppetDB to get a set of all resources regex matching the specified namespace
    # applied to a node.
    #
    #
    # Example:
    #
    #   resources_applied_to_node('foo.com', 'Puppet_Enterprise::', 'production')
    #
    #   returns
    #
    #   {
    #     'puppet_enterprise::puppet_master_host' => 'foo.com',
    #     ...
    #     'puppet_enterprise::profile::master::foo' => 'bar',
    #   }
    #
    # @param node [String] The certificate name of the node
    # @param namespace [String] The puppet resource namespace to search for
    #
    # @return [Hash] A mapping of the fully qualified param to value
    def resources_applied_to_node(node, namespace)
      resources = Puppet::Util::Pe_conf::Recover::Puppetdb.get_resources_for_node(node, namespace)
      classification = {}
      resources.each do |resource|
        params = resource['parameters']
        title = resource['title']

        next if params.empty?

        params.each do |k,v|
          full_param = "#{title}::#{k}".downcase
          classification[full_param] = v
        end
      end
      classification
    end

    # Queries the PE Node Classifier via the api for classification information
    # on the specified node. Uses the following API endpoint to collect this information:
    #
    #   /v1/classified/nodes/<name>
    #
    # @param certname [String] The certificate name of the node
    # @param facts [Hash] The regular, non trusted facts for the node
    # @param trusted_facts [Hash] The trusted facts for the node
    #
    # @return [Hash] A mapping of the fully qualified parameter namespace and it's value
    def classifier_overrides_for_node(certname, facts, trusted_facts)
      classes = classifier.get_classification(certname, facts, trusted_facts)['classes']
      overrides = {}

      classes.each do |class_ns, params|
        next if params.empty?
        params.each do |param, value|
          overrides["#{class_ns}::#{param}"] = value
        end
      end

      overrides
    end

    # Returns the currently configured node terminus for the master run mode
    def get_node_terminus
      @node_terminus ||=
          if puppet_conf[:master] && puppet_conf[:master][:node_terminus]
            puppet_conf[:master][:node_terminus]
          else
            `/opt/puppetlabs/puppet/bin/puppet config print node_terminus --section master`.chomp
          end
    end

    # Returns the currently configured external nodes for the master run mode
    def get_external_nodes
      @external_nodes ||=
          if puppet_conf[:master] && puppet_conf[:master][:external_nodes]
            puppet_conf[:master][:external_nodes]
          else
            `/opt/puppetlabs/puppet/bin/puppet config print external_nodes --section master`.chomp
          end
    end

    # Returns true if the customer is using the PE Node Classifier by looking for
    # the `node_terminus` setting in `puppet.conf` - only useful if ran on the master node.
    def using_node_classifier?
      get_node_terminus == 'classifier'
    end

    # internal helper accessors for talking to Hiera and the NC
    private

    # Creates an instance of a PuppetX::Puppetlabs::Meep::HieraAdapter for use during lookups of customers
    # information
    def hiera_adapter
      @hiera_adapter ||= PuppetX::Puppetlabs::Meep::HieraAdapter.new(Puppet.settings[:hiera_config])
    end

    def classifier
      @classifier ||= Puppet::Util::Pe_node_groups.new
    end

    # loads a puppet.conf file into a hash for lookups.
    # This is needed due to the way Puppet settings are loaded based
    # on run_mode and the node_terminus setting being set in a non default
    # section.
    # @param puppet_conf_path [String] location of puppet.conf to parse
    # @return [Hash] A hash of the puppet settings, or failing that an empty hash
    # @api private
    def puppet_conf(puppet_conf_path = '/etc/puppetlabs/puppet/puppet.conf')
      unless @puppet_conf
        if File.exists?(puppet_conf_path)
          Puppet.debug(_('Loading puppet.conf'))
          conf_sections = Puppet.settings.parse_file(puppet_conf_path).to_h[:sections]
          @puppet_conf = conf_sections.each_with_object({}) do |section, out_hash|
            section_name, section_settings = section
            out_hash[section_name] ||= {}
            section_settings.settings.each do |setting|
              out_hash[section_name][setting.name] = setting.value
            end
          end
        else
          Puppet.debug(_('puppet.conf is not located in the default location'))
          @puppet_conf = {}
        end
      end

      @puppet_conf
    end

    # Creates a hash that can be passed to hiera lookup as the scope.
    #
    # A hiera scope is essentially a set of facts that will be used during
    # lookups to traverse the various levels of the hierarchy.
    #
    # From testing, it appears the facts need to be duplicated and `::`
    # prepended to each fact name in order for any hieracrchy that uses
    # that notation to denote top scope.
    #
    # https://docs.puppet.com/hiera/3.2/command_line.html#command-line-variables
    # https://docs.puppet.com/hiera/3.2/command_line.html#json-and-yaml-scopes
    #
    # From that second link,
    #
    # Note: For Puppet, facts are top-scope variables, so their fully-qualified
    # form is $::fact_name. When called from within Puppet, Hiera will
    # correctly interpolate %{::fact_name}. However, Facter’s command-line
    # output doesn’t follow this convention — top-level facts are simply called
    # fact_name. That means you’ll run into trouble in this section if you have
    # %{::fact_name} in your hierarchy.
    #
    # @param facts       [Hash] A hash containing a nodes facts
    # @param environment [String] The puppet environment to perform the lookup in.
    def create_hiera_scope(facts, environment)
      fact_hash = facts.merge({'environment' => environment})

      # duplicate each fact to include top scope, this is needed for
      # hiera to properly match any hierachies that are defined with
      # `::` to denote top scope.
      facts.each do |fact, value|
        fact_hash["::#{fact}"] = value
      end
      fact_hash
    end
  end
end
