require 'puppet/util/execution'
require 'puppet/face'
require 'json'

module PuppetX
module Puppetlabs
module Meep
module Configure
  # Utility for making very simple PostgreSQL queries through the local
  # /opt/puppetlabs/server/bin/psql tool.
  #
  # The {PSQL#run()} function is being used to obtain a few
  # values from SHOW and version() necessary for upgrades.
  #
  # The {PSQL#query()} function can be used for obtaining a Hash of
  # rows from simple queries. But it is not intended as a general purpose
  # tool for making database queries.
  #
  # Relies on Puppet::Util::Execution to do the lifting.
  class PSQL

    # Override for tests.
    def self.psql_bin
      '/opt/puppetlabs/server/bin/psql'
    end

    ENCODING_QUERIES = %w{
      server_encoding
      lc_ctype
      lc_collate
    }.freeze

    attr_accessor :uid, :user, :password, :sslcert, :sslkey, :sslmode, :sslcacert, :database, :host, :port

    def initialize(options = {})
      options.each do |k,v|
        attr  = "#{k}="
        self.send(attr, v) if self.respond_to?(attr)
      end
    end

    def command(query, expanded: false)
      cmd = []
      # Reference: https://www.postgresql.org/docs/9.6/static/app-psql.html
      cmd << self.class.psql_bin
      # -q/--quiet Specifies that psql should do its work quietly.
      cmd << %Q{-q}

      if expanded
        cmd << %Q{--no-align}
        cmd << %Q{--expanded}
      else
        # In tuples-only mode, only actual table data is shown.
        cmd << %Q{-P}
        cmd << %Q{tuples_only=on}
      end

      # -X, --no-psqlrc Do not read the start-up file (neither the system-wide psqlrc file nor the user's ~/.psqlrc file).
      cmd << %Q{-X}
      cmd << %Q{--username=#{user}} if user
      cmd << %Q{--host=#{host}} if host
      cmd << %Q{--port=#{port}} if port
      cmd << %Q{--dbname=#{database}} if database
      cmd << %Q{--command=#{query}}
    end

    # @return [Hash] of the server_encoding, lc_ctype and lc_collate values
    #   from the Postgresql server or cached file.
    def encoding_data
      psql_data_filepath = "/opt/puppetlabs/installer/psqlinfo.json"
      if File.exist?(psql_data_filepath)
        JSON.parse(File.read(psql_data_filepath))
      else
        ENCODING_QUERIES.inject({}) do |hash,key|
          hash[key] = run("SHOW #{key}")
          hash
        end
      end
    end

    # @return the x.y.z of the Postgres server
    def server_version
      # PostgreSQL 9.6.3 on x86_64-pc-linux-gnu, compiled by gcc (GCC) 4.4.7 20120313 (Red Hat 4.4.7-4), 64-bit
      # PostgreSQL 11.5 on x86_64-pc-linux-gnu, compiled by gcc (GCC) 4.8.5 20150623 (Red Hat 4.8.5-36), 64-bit
      output = run("select version();")
      md = output.match(/PostgreSQL (\d+\.\d+(?:\.\d+)?)/)
      md[1] if md
    end

    # @param query [String] a Postgresql query expected to return parseable output.
    # @param expanded [Boolean] whether to return rows with headers, unaligned
    # and delimited for parsing.
    # @param combine [Boolean] whether to combine stderr into the execution output stream.
    # @return [String] the result of the query.
    # @raise Puppet::ExecutionFailure if the command fails to execute.
    def run(query, expanded: false, combine: false)
      psql_command = command(query, expanded: expanded)

      options = {
        :failonfail => true,
      }
      options[:combine] = true if combine
      if uid
        options[:uid] = uid
        options[:cwd] = '/opt/puppetlabs/server/'
      end

      # The provided uid may not have access to the pwd.
      options[:custom_environment] = {}

      # Reduce the logging level so we don't get DEBUG, NOTICE or WARNING messages
      # Reference: https://www.postgresql.org/docs/9.6/static/runtime-config-logging.html
      options[:custom_environment]["PGOPTIONS"] = '--client-min-messages=error'

      options[:custom_environment]["PGPASSWORD"] = password if password

      # Cert based auth requires the client cert and key, as well as the CA cert
      options[:custom_environment]["PGSSLCERT"] = sslcert if sslcert
      options[:custom_environment]["PGSSLKEY"] = sslkey if sslkey
      options[:custom_environment]["PGSSLMODE"] = sslmode if sslmode
      options[:custom_environment]["PGSSLROOTCERT"] = sslcacert if sslcacert

      result = Puppet::Util::Execution.execute(psql_command, options)
      # Strip the output, and return as a String, not a ProcessOutput, because
      # with failonfail set true, a nonzero exit will instead have raised an
      # ExecutionFailure
      result.to_s.strip
    end

    # Example:
    #   PSQL.query('select * from sometable')
    #
    #   [
    #     {
    #       'thing' => 'cat',
    #       'id'    => '1',
    #       ...
    #     },
    #     {
    #       'thing' => 'dog',
    #       'id'    => '2',
    #       ...
    #     },
    #     ...
    #   ]
    #
    # Bear in mind that nothing is typed, all output will be strings.
    #
    # @param select [String] a Postgresql SELECT query expected to return parseable
    # rows.
    # @return [Array<Hash>] returns an Array of rows, each row represented by a Hash
    # of column data.
    # @raise Puppet::ExecutFailure if the command fails.
    def query(select)
      result = run(select, expanded: true, combine: true)
      records = result.split(/\n\n/)
      records.map do |r|
        fields = r.split("\n")
        fields.each_with_object({}) do |f,hash|
          col, val = f.split('|')
          hash[col] = val
        end
      end
    end
  end
end
end
end
end
