module PuppetX
module Puppetlabs
module Meep
module Configure
  # These helpers rely on the format of the puppetlabs-pe_install pe_postgresql_info
  # structured fact, which is returned as a hash of values detailing the installed
  # server version, data directories, partition and used disk space.
  module PostgresHelpers
    def pe_postgresql_info
      state[:pe_postgresql_info] || {}
    end

    def available_bytes
      pe_postgresql_info.fetch('data_partition_available_bytes', 0).to_i
    end

    # @param version [String] Expects a Postgresql major.minor version string
    #   (EX 9.4)
    def used_bytes_for(version)
      the_version(version).fetch('used_bytes', 0).to_i
    end


    # give ourselves a 110% buffer for other processes and additional PE
    # files that may be installed/copied during the upgrade
    def required_bytes_for(version)
      (used_bytes_for(version) * 1.10).round
    end

    def the_version(version)
      versions = pe_postgresql_info['versions'] || {}
      versions[version] || {}
    end

    def pg_data_dir(version)
      the_version(version)['data_dir']
    end

    def postgresql_not_installed?
      version = pe_postgresql_info['installed_server_version']
      version.nil? || version.empty?
    end

    def postgresql_installed?
      !postgresql_not_installed?
    end
  end

  class ValidationMessage
    attr_accessor :severity, :message
    def initialize(severity, message)
      @severity = severity
      @message = message
    end
  end

  class ValidationRule
    attr_accessor :state

    def initialize(state)
      self.state = state
    end

    def valid?
      raise(NotImplemented, _('ValidationRule must implement valid?().'))
    end

    def messages
      messages = []
      if !valid?
        messages = _messages
      end
      messages
    end

    def _messages
      raise(NotImplemented, _('ValidationRule must implement _messages().'))
    end
  end

  class PostgresMigrationSpace < ValidationRule
    include PostgresHelpers

    def valid?
      if !state[:postgres_migration] || postgresql_not_installed?
        return true
      end

      previous_version = state[:previous_postgres_version]

      (available_bytes - required_bytes_for(previous_version)) >= 0
    end

    def _messages
      previous_version = state[:previous_postgres_version]

      m = [
        _("Insufficient space to migrate the Postgresql database in %{data_dir}.") % {
          :data_dir => pg_data_dir(previous_version),
        },
        _("Available bytes: %{available_bytes}. Required bytes: %{required_bytes}.") % {
          :available_bytes => available_bytes,
          :required_bytes => required_bytes_for(previous_version),
        },
        _("Because we keep a backup of your %{previous_version} database, we need as much free space as your %{previous_version} database currently uses to continue.") % {
          :previous_version => previous_version,
        },
        _("If you wish to force the upgrade to proceed, you may re-run the original puppet-infrastructure configure command with the --force flag, but pg_upgrade will fail if there is insufficient space."),
      ]

      [ValidationMessage.new(:fatal, m.join("\n"))]
    end
  end

  class ReplicaUpgrade < ValidationRule
    include PostgresHelpers
    def valid?
      state[:replica_pg_installed] 
    end

    def _messages
      m = [
        _("There were problems with the attempted replica upgrade.")
      ]

      if !valid?
        m << _("Host %{host} does not appear to be a replica. Please use this command with an enabled replica.") % { :host => state[:replica_host] }
      end

      [ValidationMessage.new(:fatal, m.join("\n"))]
    end
  end

  class CompilerUpgrade < ValidationRule
    def valid?
      ['compiler', 'pe_compiler'].include?(state[:compiler_type])
    end

    def _messages
      m = [
        _("There were problems with the attempted compiler upgrade.")
      ]

      if !valid?
        m << _("Host %{host} does not appear to be a compiler. Please use this command with a valid compiler node.") % { :host => state[:compiler_host] }
      end

      [ValidationMessage.new(:fatal, m.join("\n"))]
    end
  end
    

  class ExternalPostgres < ValidationRule
    include PostgresHelpers

    def valid?
      if !state[:separate_database_node] || postgresql_installed? || !state[:upgrade_from]
        return true
      end

      external_version = state[:external_database_version]
      version_is_behind = state[:external_database_behind]

      !(external_version.nil? || version_is_behind)
    end

    def _messages
      database_host = state[:database_host]
      external_version = state[:external_database_version]

      m = [
        _("Your installation is managing the Postgresql database on a separate host (%{database_host}).") % { :database_host => database_host },
      ]

      if external_version.nil?
        m << _("We were unable to confirm the version of Postgresql on %{database_host}.") % { :database_host => database_host }
        m << _("Please review the logs for connectivity issues.")
        m << _("If you wish to force the upgrade to proceed, you may re-run the original puppet-infrastructure configure command with the --force flag, but Puppet Enterprise will experience issues if Postgresql is not the correct version.")
      else
        m << _("We have detected that %{database_host} is running Postgresql %{database_version}.") % { :database_host => database_host, :database_version => external_version }
      end

      m << _("Puppet Enterprise requires Postgresql %{new_version}, please upgrade your %{database_host} host's Postgresql instance before continuing with the rest of the upgrade of Puppet Enterprise.") % { :new_version => state[:new_postgres_version], :database_host => database_host }

      [ValidationMessage.new(:fatal, m.join("\n"))]
    end
  end

  class ParameterValidation < ValidationRule
    def valid?
        if state[:jruby_9k_enabled] != false
            return true
        end
    end
    def _messages
      m = [
          _("One or more currently defined parameters are invalid.")
      ]
      if !state[:jruby_9k_enabled]
          m << _("puppet_enterprise::master::puppetserver::jruby_9k_enabled is set to false. You can no longer use JRuby 1.7. Please update your PE install to use JRuby 9k.")
      end
      [ValidationMessage.new(:fatal, m.join("\n"))]
    end
  end

  # The Validator is a simple class that is initialized with a Hash of state.
  # After initialization {Validator.register} creates some set of
  # ValidationRules.
  #
  # The only shared state is the initial {Validator#state} Hash.
  #
  # All expensive operations should be performed before the Validator is
  # initialized, and refined data passed in through {Validator#state}. Given
  # that ValidationRule tests are inexpensive, no additional cacheing is necessary.
  #
  # The Validator will log messages through Puppet, but only if
  # {Validator.log_and_fail!} is called. An {InstallationInvalid} error will
  # only be raised if some state is invalid, and state[:force] is not true.
  class Validator 
    class InstallationInvalid < RuntimeError; end

    attr_accessor :state, :rules

    # @param state [Hash] Expects the following entries:
    #
    #   Postgresql:
    #
    #   :postgres_migration [Boolean] true if performing the postgres migration
    #   :pe_postgresql_info [Hash] the pe_postgresql_info fact
    #   :previous_postgres_version [String] the major.minor of Postgresql that
    #     we are migrating from
    #   :new_postgres_version [String] the major.minor of Postgresql that we are
    #     migrating to
    #   :separate_database_node [Boolean] true if database host is not one of the
    #     primary master, console, puppetdb nodes
    #   :database_host [String] certname of the database host, may be nil
    #   :external_database_version [String] the major.minor.patch of Postgresql that
    #     was detected on the separate database node
    #   :external_database_behind [Boolean] true if the version of Postgresql
    #     on the sepaarate database node is less than new_postgres_version
    #
    #   General:
    #
    #   :force [Boolean] true if we should ignore fatal validation errors
    #   :upgrade_from [String] version of pe we're upgrading from
    def initialize(state)
      @state = state
      @rules = []
    end

    def register(rules)
      rules.each do |r|
        raise(RuntimeError, _("Expected a ValidationRule, got '%{class}'") % { class: r }) unless r.kind_of?(Class) && r <= (ValidationRule)
        self.rules << r.new(state)
      end
    end

    def valid?
      rules.all? do |r|
        r.valid?
      end
    end

    def messages
      rules.collect do |r|
        r.messages
      end.flatten
    end

    # Log all messages at the appropriate level and then raise a Validator::Error
    # if any were fatal.
    #
    # @note Does not raise an exception if the state[:force] is true.
    # @raise PuppetX::Puppetlabs::Meep::Validator::InstallationInvalid if there are
    #   any fatal messages logged.
    def log_and_fail!
      die = false
      messages.each do |m|
        case m.severity
        when :fatal
          die = true
          Puppet.err(m.message)
        when :warning
          Puppet.warning(m.message)
        when :debug
          Puppet.debug(m.message)
        else
          Puppet.notice(m.message)
        end
      end
      raise(InstallationInvalid, _('Failed to validate the state of the current installation.')) if die && !state[:force]
    end
  end
end
end
end
end
