require 'puppet/util/pe_node_groups'
require 'puppet_x/puppetlabs/meep/util'
require 'puppet_x/util/service_status'
require 'puppet_x/util/rbac'
require 'puppet_x/util/bolt'
require 'puppet_x/util/classification'
require 'puppet_x/util/docs'
require 'puppet_x/util/infrastructure_error'
require 'puppet_x/util/orchestrator'
require 'puppet_x/util/code_manager'
require 'puppet_x/util/stringformatter'

module PuppetX
module Puppetlabs
module Meep
module Provision

  # Original provisioning workflow relying solely on puppetdbsync.
  class Base
    include Puppet::Util::Colors
    include PuppetX::Puppetlabs::Meep::Util

    # The certificate name for the replica we are provisioning.
    attr_accessor :replica_certname

    # The options hash from the provision action.
    attr_accessor :options

    # The Puppet primary certificate name.
    attr_accessor :primary_certname

    # The array of PE services from /etc/puppetlabs/client/services.conf
    attr_accessor :config

    # The array of PE node roles from /etc/puppetlabs/client/services.conf
    attr_accessor :nodes_config

    # Hash of PE service information for service found running on the
    # +replica_certname+ host.
    attr_accessor :services

    # The RBAC token for accessing PE services.
    attr_accessor :rbac_token

    def initialize(replica_certname, options, config: nil, nodes_config: nil, rbac_token: nil)
      self.replica_certname = replica_certname
      self.options = options
      self.primary_certname = Puppet.settings[:certname].strip
      self.config = config
      self.nodes_config = nodes_config
      self.rbac_token = rbac_token
    end

    def check_flags
      if options[:enable]
        Puppet.err(_('The --enable flag may only be used with the --streaming flag.'))
        exit(1)
      end
    end

    # @return A Puppet::Util::Pe_node_groups instance initialized with :classifier
    # service information from the +services+ hash. Used to interact with
    # the classifier service API.
    def classifier
      unless @classifier
        nc_service = services[:classifier]
        @classifier = Puppet::Util::Pe_node_groups.new(nc_service[:server], nc_service[:port].to_i, "/#{nc_service[:prefix]}")
      end
      return @classifier
    end

    # Caches the UUIDs of each group/environment in the classifier. If
    # the given name can't be found, it attemps to refetch the UUIDs in
    # case classification has been updated with new groups/environments.
    # @return The UUID of the group/environment name. 
    def id(name)
      if @group_ids.nil? || @group_ids[name].nil?
        @group_ids = fetch_ids
      end
      @group_ids[name]
    end

    def fetch_ids
      all_groups = PuppetX::Util::Classifier.get_groups(classifier)
      all_groups.map { |group| [group['name'], group['id']] }.to_h
    end

    def provisioning_timestamp
      unless @provisioning_ts
        @provisioning_ts = time_stamp
      end
      @provisioning_ts
    end

    # Load the config, nodes_config and services data.
    def prep_config
      self.config ||= PuppetX::Util::ServiceStatus.load_services_config()
      self.nodes_config ||= PuppetX::Util::ServiceStatus.load_nodes_config()
      self.services ||= PuppetX::Util::ServiceStatus.validate_and_select_services_from_config(config, 'provision', primary_certname)
      self.rbac_token ||= PuppetX::Util::RBAC.load_token(options[:token_file])
    end

    # Validate the primary and replica hosts.
    def validate
      orch = services[:orchestrator]
      unless options[:force]
        services_statuses = PuppetX::Util::ServiceStatus.services_for(primary_certname, nil, config, 5, nodes_config: nodes_config)
        PuppetX::Util::ServiceStatus.ensure_all_services_running(services_statuses)
        fs_status = services_statuses.find {|s| s[:service] == "file-sync-storage-service" }
        if fs_status[:status]["repos"]["puppet-code"]["latest_commit"].nil?
          Puppet.err(_("No commits have been made to puppet-code."))
          Puppet.err(_("Deploy with code-manager before provisioning by running `puppet code deploy <environment>`."))
          exit(1)
        end

        if replica_certname == primary_certname
          Puppet.err(_("%{certname} is already provisioned as the primary.") % { certname: replica_certname })
          exit(1)
        end

        existing_role = PuppetX::Util::ServiceStatus.node_role(nodes_config, replica_certname)
        if existing_role && existing_role != 'primary_master_replica'
          Puppet.err(_("%{certname} is already provisioned with the role %{role}.") % { certname: replica_certname, role: existing_role })
          exit(1)
        end

        unless PuppetX::Util::Orchestrator.node_in_orch_inventory(orch, rbac_token, replica_certname)
          Puppet.err(_("The node %{certname} is not connected to the primary via PCP.") % { certname: replica_certname } + " " +
                     _("Check that you are using the certname of the replica node, which may be different from the hostname.") + " " +
                     _("Also verify that orchestration services are enabled on the primary and that the PXP agent is running on %{certname}.") % { certname: replica_certname })
          exit(1)
        end
      end

      return true
    end

    # Ensure that the PE HA Master and PE HA Replica groups exist.
    # Ensure that the PE HA Master node group has the proposed replica
    # added so that permissions will be modified on the primary allowing
    # the replica access.
    def classify
      PuppetX::Util::Classification.provision_replica(classifier, primary_certname, replica_certname)

      puts _("Updated classification")
    end

    def run_puppet_on(certname, action:, description:)
      orch = services[:orchestrator]
      log = logfile(action: action, timestamp: provisioning_timestamp, description: description)
      PuppetX::Util::Orchestrator.run_puppet(orch, certname, {:nodes => [certname]}, rbac_token, report_log: log)
    end

    # Enforce configuration changes on the primary and replica
    # by running Puppet through the Orchestrator.
    def run_puppet
      run_puppet_on(primary_certname, action: 'provision_replica', description: 'first_primary_run')

      PuppetX::Util::CodeManager.kick_file_sync_ssl()
      PuppetX::Util::CodeManager.kick_file_sync_confd()

      begin
        run_puppet_on(replica_certname, action: 'provision_replica', description: 'replica_run')
      ensure
        run_puppet_on(primary_certname, action: 'provision_replica', description: 'second_primary_run')
      end
    end

    def pin_to_pe_infra_agent
      puts _("Pinning #{replica_certname} to PE Infrastructure Agent")
      PuppetX::Util::Classifier.pin_node_to_group(classifier, 'PE Infrastructure Agent', id('PE Infrastructure Agent'), replica_certname)
      run_puppet_on(replica_certname, action: 'provision_replica', description: 'pin_to_pe_infra_agent')
    end

    def unpin_from_pe_infra_agent
      puts _("Unpinning #{replica_certname} from PE Infrastructure Agent, since it should now be automatically classified.")
      PuppetX::Util::Classifier.unpin_node_from_group(classifier, 'PE Infrastructure Agent', id('PE Infrastructure Agent'), replica_certname)
    end

    # After a successful provision, inform the user and provide
    # instructions for tracking the status of the provision,
    # and subsequently enabling.
    def closing_text
      pdb_sync_link = PuppetX::Util::Docs.link(:PDB_SYNC_LINK)

      puts
      puts PuppetX::Util::String::Formatter.join_and_wrap_no_indent([
        colorize(:green, _("A replica has been provisioned.")) + " " + _("Services are syncing.") + " " + _("If you have a large PuppetDB instance, consult the documentation at %{link} to learn approaches for speeding up the process.\n") % { link: pdb_sync_link },
        _("To track sync progress, enter the following command:\n"),
        "`puppet infrastructure status --host %{certname}`\n" % { certname: replica_certname },
        _("Once the sync is complete, enter `puppet infrastructure enable replica %{certname}` on the primary so that it will be ready to handle a failover from the primary.\n") % { certname: replica_certname },
        _("If you run this command before the replica has fully synchronized, the replica will not have all the necessary information to replace the primary during a failover.")])
    end

    def run
      check_flags
      prep_config
      validate
      pin_to_pe_infra_agent
      classify
      run_puppet
      unpin_from_pe_infra_agent
      closing_text
    end
  end

end
end
end
end
