module PuppetX
  module Puppetlabs
    module Meep
      module PuppetContext

        # Invokes the passed block in a Puppet context prepared with new environment
        # loader and settings. Environment settings, loader and current_environment
        # will be temporarily reset based on the passed environmentpath, environment
        # and basemodulepath. Any additional puppet settings passed in the
        # terminal hash will be temporarily reset to the given values as well.
        #
        # @param environmentpath [String] Temporary environmentpath.
        # @param environment [String] Temporary environment name.
        # @param basemodulepath [String] Temporary basemodulepath (':' delimited).
        #   (Default is an empty string.)
        # @param additional_settings_to_override [Hash<Symbol,Object>] Hash of
        #   other puppet settings (in addition to environment, environmentpath,
        #   basemodulepath) to temporarily replace during the block's execution.
        def puppet_environment_context(environmentpath, environment, basemodulepath = '', additional_settings_to_override = {}, &block)
          settings_to_override = {
            :environmentpath => environmentpath,
            :environment     => environment,
            :basemodulepath  => basemodulepath,
            :manifest        => '/dev/null',
          }
          settings_to_override = additional_settings_to_override.merge(settings_to_override)

          original_settings = override_puppet_settings(settings_to_override)

          loader = Puppet::Environments::Directories.new(environmentpath, basemodulepath.split(':'))

          Puppet.override(
            :environments => loader,
          ) do
            environment = loader.get(environment)

            Puppet.override(
              :current_environment => environment
            ) do
              block.call
            end

          end

        ensure
          restore_puppet_settings(original_settings)
        end

        private

        # @param settings_to_override [Hash] of key value pairs of Puppet settings
        #   we're going to change.
        # @return [Hash] of the original setting values that we changed.
        def override_puppet_settings(settings_to_override)
          original_settings = {}
          settings_to_override.each do |k,v|
            original_settings[k] = Puppet[k]
            Puppet[k] = v
          end
          original_settings
        end

        def restore_puppet_settings(original_settings)
          original_settings.each do |k,v|
            Puppet[k] = v
          end
        end
      end
    end
  end
end
