require 'puppet/util/colors'
require 'fileutils'

module PuppetX
  module Util
    class CodeManager
      # file sync needs to see something actually change in order to create an
      # initial commit when it auto-commit mode; touch a file in the ca dir to get
      # this to happen.
      def self.kick_file_sync_ssl()
        FileUtils.touch('/etc/puppetlabs/puppet/ssl/ca/ca_pub.pem')
      end

      def self.kick_file_sync_confd()
        FileUtils.touch('/etc/puppetlabs/enterprise/conf.d/pe.conf')
      end

      def self.copy_client_data_to_storage_service(
        codedir,
        file_sync_config = '/etc/puppetlabs/puppetserver/conf.d/file-sync.conf',
        data_dir = '/opt/puppetlabs/server/data/puppetserver/filesync/',
        user = 'pe-puppet',
        group = user)

        code_staging = '/etc/puppetlabs/code-staging'
        if File.exist?(file_sync_config)
          config = Hocon.load(file_sync_config)
          val = config.dig('file-sync', 'repos', 'puppet-code', 'staging-dir')
          code_staging = val if val
        end

        FileUtils.rm_rf(code_staging)
        FileUtils.cp_r(codedir, code_staging)

        # With versioned-dirs we turn each environment into a symlink to somewhere
        # in opt. By default FileUtils.cp_r dereferences a top level symlink if
        # given one as a source, however it will keep lowerlevel symlinks. So, if
        # we're in versioned-dirs mode we need to copy each environment to overwrite
        # the symlink with the actual contents.
        environments = Dir.glob("#{codedir}/environments/*")
        environments.each do |env|
          if File.symlink?(env)
            env_name = File.basename(env)
            staging_env = "#{code_staging}/environments/#{env_name}"
            FileUtils.rm_rf(staging_env)
            FileUtils.cp_r(env, staging_env)
          end
        end

        storage_dir = "#{data_dir}/storage"
        FileUtils.chown_R(user, group, code_staging)
        FileUtils.rm_rf(storage_dir)
        FileUtils.mkdir(storage_dir)

        client_files = Dir.glob("#{data_dir}/client/*")
        client_only_dirs = %w{worktrees versioned-dirs}
        files_to_copy = client_files.reject { |f| client_only_dirs.include? File.basename(f) }
        files_to_copy.each do |file|
          FileUtils.cp_r(file, storage_dir + '/')
        end
        FileUtils.chown_R(user, group, storage_dir)

        # Versioned-dirs causes the HEAD file in modules to be set to the actual git
        # SHA rather than a branch reference. This is fine for clients but breaks
        # the way that we commit on the storage service.
        submodule_dir = "#{storage_dir}/puppet-code"
        environments = Dir.glob("#{submodule_dir}/*")
        head_content = "ref: refs/heads/master\n"
        environments.each do |env|
          env_name = File.basename(env)
          head_path = "#{submodule_dir}/#{env_name}/HEAD"
          File.write(head_path, head_content)
        end
        puppet_code_head = "#{storage_dir}/puppet-code.git/HEAD"
        File.write(puppet_code_head, head_content)
      end
    end
  end
end
