require 'digest/sha1'
require 'puppet/error'
require 'puppet/agent'
require 'puppet_x/util/stringformatter'
require 'puppet_x/util/apply'

module PuppetX
  module Util
    class Forget

      def self.pglogical_hash(certname)
        Digest::SHA1.hexdigest(certname).slice(0, 14)
      end

      def self.generate_dropping_manifest(master_certname, replica_certname)
        primary_node = 'n' + pglogical_hash(master_certname)
        replica_sub = 's' + pglogical_hash(replica_certname)
        slots = %w(pe_classifier pe_activity pe_rbac pe_orchestrator pe_inventory)
                .map { |db| "pgl_#{db}_#{primary_node}_#{replica_sub}" }
        resources = slots
                    .map { |slot| "pe_postgresql_replication_slot {'#{slot}': ensure => absent}"}
                    .join("\n")
        "include puppet_enterprise::params\n" + resources
      end

      def self.ensure_replication_slots_absent(master_certname, replica_certname)
        manifest = generate_dropping_manifest(master_certname, replica_certname)
        Puppet.debug(_('Applying the following Puppet manifest...'))
        Puppet.debug(manifest)
        PuppetX::Util::Apply.run_puppet_apply(manifest, [0,1,2,4,6])
      end

      def self.drop_replication_slots_or_exit(master_certname, replica_certname)
        case ensure_replication_slots_absent(master_certname, replica_certname)
        when :unchanged then puts _("No replication slots found for %{certname}. Continuing.") % { certname: replica_certname }
        when :changed then puts _("Removed replication slots for %{certname}.") % { certname: replica_certname }
        when :failure then
          puts PuppetX::Util::String::Formatter.wrap_no_indent(
            _("Failed to remove replication slots for %{certname}.") % { certname: replica_certname } + " " +
            _("Ensure that the replica node is off the network, or that the pe-postgresql service is not running, before attempting to forget the replica."))
          exit 1
        end
      end
    end
  end
end
