require 'set'
require 'puppet_x/util/stringformatter'
require 'io/console'

module PuppetX::Util
  class Interview
    YES_REGEX = Regexp.new('^y(es)?$', Regexp::IGNORECASE)
    NO_REGEX = Regexp.new('^n(o)?$', Regexp::IGNORECASE)
    SERVER_REGEX = /^[a-z0-9._-]+:\d+$/

    def self.confirm(question, default=true, stream=STDIN)
      puts question
      confirm = nil
      3.times do
        print "> "
        $stdout.flush
        input = stream.gets.chomp.strip
        if input == ''
          confirm = default
        elsif input =~ YES_REGEX
          confirm = true
        elsif input =~ NO_REGEX
          confirm = false
        else
          puts _("Could not parse input, answer 'y' or 'n'.")
          next
        end
        break
      end
      if confirm.nil?
        puts _("Could not parse confirmation input, exiting.")
        exit(1)
      end
      return confirm
    end

    # TODO: HA iteration 1 tech debt. This should accept a meaningful symbol to
    # return intead of a single character
    def self.ask_choices(pre_question, post_question, choices, default=:a, stream=STDIN)
      improved_choices = choices.map do |k, desc|
        { str: k == default ? k.to_s.upcase : k.to_s,
          regex: Regexp.new("^#{k.to_s}$", Regexp::IGNORECASE),
          desc: desc,
          key: k,
        }
      end
      possible_choices = improved_choices.map { |c| c[:str] }.join('/')
      possible_choices = "[#{possible_choices}]"

      puts pre_question
      improved_choices.each { |choice| puts "  #{choice[:str]}: #{choice[:desc]}" }
      puts ""
      puts "#{post_question} #{possible_choices}"
      answer = nil
      3.times do
        print "> "
        $stdout.flush
        input = stream.gets.chomp.strip
        if input == ''
          answer=default
        else
          improved_choices.each do |choice|
            if choice[:regex] =~ input
              answer = choice[:key]
            end
          end
        end
        if !answer.nil?
          break
        else
          puts _("\nCould not parse input, answer %{possible_choices}.") % { possible_choices: possible_choices }
        end
      end
      if answer.nil?
        puts _("Could not parse input, exiting.")
        exit(1)
      end
      return answer
    end

    def self.ask_password(question, regex=/.+/)
      retries = 0
      begin
        print "#{question} >"
        answer = STDIN.noecho(&:gets).chomp
        raise unless answer =~ regex
      rescue
        answer = nil
        if (retries += 1) < 3
          print _("\nInvalid input. ")
          retry
        end
      end
      if answer.nil?
        puts _("\nCould not get valid input, exiting.")
        exit 1
      end
      print "\n"
      answer
    end

    def ask_question(question, regex=/.*/, default=nil, stream=STDIN)
      print question
      print " (#{default})" if default
      print "\n"
      answer = nil
      3.times do
        input = stream.gets.chomp.strip
        if default && input == ''
          answer=default
        elsif input =~ regex
          answer = input
        else
          puts _("Could not parse input. %{question}") % { question: question }
        end
      end
      if answer.nil?
        puts _("Could not parse input, exiting.")
        exit(1)
      end
      return answer
    end

    def self.ask_server_list(question, stream=STDIN)
      puts question
      answer = nil
      3.times do
        print '> '
        $stdout.flush
        input = stream.gets.chomp.strip
        begin
          answer = parse_server_list(input)
          return answer
        rescue RuntimeError => e
          puts e.message
          puts
          puts question
        end
      end
      if answer.nil?
        puts _("Could not parse input, exiting.")
        exit(1)
      end
      return answer
    end

    def self.get_topo(options)
      topo_choices = { a: _('Monolithic with a single primary'),
                       b: _('Monolithic with compilers'),
                     }
      if options[:topology]
        case options[:topology]
        when 'mono'
          topo = :a
        when 'mono-with-compile'
          topo = :b
        else
          # Workaround FM-6292
          # rubocop:disable GetText/DecorateFunctionMessage
          raise PuppetX::Util::InfrastructureUnknownTopologyError.new(topo, ["mono", "mono-with-compile"])
        end
      else
        ask_choices(_('Describe your PE setup'),
                    _('Which describes your setup?'),
                    topo_choices)
      end
    end
    
    def self.promote_interview(conf, master_certname, replica_certname, options)
      unless STDIN.tty?
        if options[:topology].nil? || options[:yes].nil?
          puts _("The promote interview requires a tty to proceed.") + " " + _("To promote without an interview select your topology and pass the --topology and --yes flags to `puppet infrastructure promote`.")
          exit(1)
        end
      end

      unless options[:yes]
        unless PuppetX::Util::Interview.confirm(_('Make sure that the primary is offline and cannot come back.') + " " + _('Is the primary offline?') + '  [Y/n]', true)
          puts _("Check the primary to see if it's working or can be fixed.") + " " + _("If it has stopped working and cannot be brought back online, run `puppet infrastructure promote replica` to replace the primary with the replica.")
          exit(0)
        end
      end

      topo = get_topo(options)
      configure_agent = !options[:skip_agent_config]
      puts
      puts _("This is your current configuration:")
      puts PuppetX::Util::Classification.config_output(conf, configure_agent)
      suggestions = PuppetX::Util::Classification.suggest_promote_config(conf, topo, master_certname, replica_certname, configure_agent)
      case topo
      when :a
        if options[:agent_server_urls] || options[:pcp_brokers] || options[:primary_uris]
          raise _("--agent-server-urls, --pcp_brokers, and --primary-uris cannot be passed when enabling the 'mono' topology.") + " " + _("Use 'mono-with-compile'")
        end
      when :b
        if options[:agent_server_urls]
          suggestions[:agent_list] = parse_server_list(options[:agent_server_urls])
        end
        if options[:pcp_brokers]
          suggestions[:pcp_broker_list] = parse_server_list(options[:pcp_brokers])
        end
        if options[:primary_uris]
          suggestions[:primary_uris_list] = parse_server_list(options[:primary_uris])
        else
          # Allow the user to just pass in --agent-server-urls and --pcp-brokers, since
          # primary_uris should match server_list.
          suggestions[:primary_uris_list] = suggestions[:agent_list]
        end
      end

      puts
      puts _("Once this replica is promoted, your new configuration will be:")
      puts PuppetX::Util::Classification.config_output(suggestions, configure_agent)
      unless options[:yes]
        puts
        unless confirm(_("Use the new configuration and continue promoting the replica by running puppet on all infrastructure nodes?") + " [Y/n]")
          exit(1)
        end
      end
      return suggestions
    end

    CONFIG_FLAG_MESSAGES = [
        [:infra_agent_server_urls, _("Hostnames and ports your Puppet Servers to run your Puppet Infrastructure Agents against")],
        [:agent_server_urls, _("Hostnames and ports of your Puppet Servers to run your Puppet Agents against")],
        [:classifier_termini, _("Hostnames and ports of your Node Classifiers")],
        [:puppetdb_termini, _("Hostnames and ports of your PuppetDBs")],
        [:pcp_brokers, _("Hostnames and ports of your PCP Brokers")],
        [:infra_pcp_brokers, _("Hostnames and ports of your PCP Brokers for Puppet Infrastructure")],
        [:primary_uris, _("Hostnames and ports of your PE Primaries to connect PXP agents to")],
        [:infra_primary_uris, _("Hostnames and ports your PE Primaries to connect your Puppet Infrastructure PXP agents against")],
    ]

    def self.any_custom_flags?(options)
      infra_flags = Set.new([:infra_agent_server_urls, :classifier_termini, :puppetdb_termini, :infra_pcp_brokers, :infra_primary_uris])
      option_flags = Set.new(options.keys)
      return infra_flags.intersect?(option_flags)
    end

    def self.parse_server_list(list_str)
      if list_str == ''
        raise _("Could not parse server list: should be in the form 'example.com:8080'")
      end
      list = list_str.split(',')
      list.each do |server|
        unless server =~ SERVER_REGEX
          raise _("Could not parse server list: '%{server}' should be in the form 'example.com:8080'") % { server: server }
        end
      end
      list
    end

    # TODO: tech debt We should validate after parsing so we can validate after
    # generating/interviewing as well.
    def self.validate_infra_flags(action, options)
      flags_missing_values = []
      # We allow --primary-uris and/or --infra-primary-uris to not be specified, since
      # they should take the value of --agent-server-urls and --infra-agent-server-urls
      # by default.
      only_missing_allowed_flags = true
      CONFIG_FLAG_MESSAGES.each do |flag, message|
        unless options[flag]
          flags_missing_values << "--#{flag.to_s.gsub(/_/,'-')} #{message}\n"
          only_missing_allowed_flags &= [:primary_uris, :infra_primary_uris].include?(flag)
        end
      end

      unless flags_missing_values.empty? || only_missing_allowed_flags
        error_string = _("Cannot %{action} a replica without the following flags:") % { action: action }
        error_string << "\n"
        error_string << flags_missing_values.join(" ") << "\n"
        raise error_string
      end
    end

    def self.parse_infra_flags(options)
        {
          agent_list: parse_server_list(options[:agent_server_urls]),
          infra_agent_list: parse_server_list(options[:infra_agent_server_urls]),
          pcp_broker_list: parse_server_list(options[:pcp_brokers]),
          infra_pcp_broker_list: parse_server_list(options[:infra_pcp_brokers]),
          primary_uris_list: options[:primary_uris] ? parse_server_list(options[:primary_uris]) : parse_server_list(options[:agent_server_urls]),
          infra_primary_uris_list: options[:infra_primary_uris] ? parse_server_list(options[:infra_primary_uris]) : parse_server_list(options[:infra_agent_server_urls]),
          puppetdb_list: parse_server_list(options[:puppetdb_termini]),
          classifier_list: parse_server_list(options[:classifier_termini]),
        }
    end

  end
end
