require 'json'
require 'puppet_x/util/orchestrator/runner'
require 'puppet_x/util/stringformatter'

module PuppetX
  module Util
    class Orchestrator

      # Handles Orchestrator:
      #
      # /orchestrator/v1/command/plan_run
      #
      # and unpacking the events from a:
      #
      # /orchestrator/v1/plan_job/#
      #
      # with the help of PlanJob and Job classes.
      class PlanRunner < Runner

        attr_accessor :log

        def initialize(service_config, rbac_token, log)
          super(service_config, rbac_token)
          self.log = log
          @command_type = 'plan'
        end

        def start_command(plan:, description:, params:, environment:, **_kwargs)
          url = "#{service_config[:url]}/v1/command/plan_run"
          payload = {:environment => environment, :plan_name => plan, :description => description, :params => params}
          uri = URI(url)
          response = orch_connection.post(uri, payload.to_json, headers: orch_request_headers)
          check_orch_response(response, 202)
          job = JSON.parse(response.body)

          plan_job_for(job['name'])
        end

        # Plans tend to have many steps. For parity with Bolt, and puppet-infra
        # plan output when running with bolt, the PlanRunner.run() will display
        # events as they update rather than wait for the plan_job to finish and then
        # dump status the way we do for Puppet/Task runs.
        def _wait_for_orch_job(job:, plan:, **_kwargs)
          puts
          notification = [
            _("Running %{plan} plan.") % { plan: plan },
          ]
          puts notification.join(' ')
          puts
          puts colorize(:green, _("Starting: plan %{name}") % { name: plan })

          # This is a plan_job we're waiting on...
          last_event = nil

          while job.loading? || job.running?
            job.wait_for_job_status(reload: true)

            if !job.loading?
              last_event = _display_events(job, last_event)
            end
            sleep(orch_status_wait_seconds)
          end
          puts colorize(:green, _("Finished: plan %{name} in %{duration}") % { name: plan, duration: _duration_to_string(job.duration) })
          job
        end

        def _display_task_progress(details)
          id = details['job-id']
          job = job_for(id)
          job.wait_for_job_completion

          type = job.type.split('_').last
          scope = job.scope
          node_clause = scope.empty? ? '' : _("on %{nodes}") % { nodes: scope.join(',') }
          messages = [
            _("Starting: %{job_type} %{job_action}") % { job_type: type, job_action: job.action },
            node_clause,
          ]
          _output(messages.join(' '), color: :green)
          _output(
            format(
              "Finished: %{job_type} %{job_action} with %{failure_count} failures in %{duration}",
              { job_type: type, job_action: job.action, failure_count: job.node_failure_count, duration: _duration_to_string(job.duration) }
            ),
            color: :green
          )
          if job.action == 'enterprise_tasks::run_puppet'
            log.puts 'Output from the above puppet run:'
            log.puts job.nodes_body['items'][0]['result']['_output']
          end

          if job.failed? && !job.task_errors.empty?
            _output(_("Errors in %{job_type} %{job_action}:") % { job_type: type, job_action: job.action }, color: :red)
            job.task_errors.each do |node_name, error|
              _output("  #{node_name}:", color: :red)
              _output(_("    Message: %{msg}") % { msg: error['msg'] }, color: :red)
              _output(_("    Kind: %{kind}") % { kind: error['kind'] }, color: :red)
              _output(_("    Details:"), color: :red)
              _output(
                PuppetX::Util::String::Formatter.indent(error['details'].pretty_inspect, 6),
                color: :red
              )
            end
          end
        end

        # Display stepwise output of a plan based on current events since the
        # +last_event+, to mimic Bolt output.
        def _display_events(plan_job, last_event)
          plan_job.events_since(last_event).each do |e|
            type = e['type']
            details = e['details']
            case type
            when %r{(?:task|wait|command)_start}
              _display_task_progress(details)
            when 'out_message'
              _output(details['message'])
            when 'plan_start' # run_plan() starts
              _output(_("Starting: plan %{plan}") % { plan: details['plan'] }, color: :green)
            when 'plan_end' # run_plan() ends
              _output(_("Finished: plan %{plan} in %{duration}") % { plan: details['plan'], duration: _duration_to_string(details['duration']) }, color: :green)
            when 'plan_failed' # plan returns an error
              log.puts("Failed plan: #{details}")
            when 'plan_finished' # end of main plan, handled by caller...
            else
              # TBD
              _output("#{type}: #{details}")
            end
          end
          plan_job.events.last
        end

        def _handle_job_failure(job)
          result = job.error
          raise PuppetX::Util::OrchestratorPlanFailedError.new(result['msg'], result['kind'], result['details']) #rubocop:disable GetText/DecorateFunctionMessage
        end

        def _handle_error(error, args)
          Puppet.err(_("Error during orchestrated plan run of %{plan}." % { plan: args[:plan] }))
          raise error
        end

        # Sends puts message to the console and the PE installer directory logging
        def _output(message, color: :white)
          puts colorize(color, message)
          log.puts message
        end

        # Lifted from Bolt::Outputter::Human
        def _duration_to_string(duration)
          return 0 if duration.nil?
          hrs = (duration / 3600).floor
          mins = ((duration % 3600) / 60).floor
          secs = (duration % 60)
          if hrs > 0
            "#{hrs} hr, #{mins} min, #{secs.round} sec"
          elsif mins > 0
            "#{mins} min, #{secs.round} sec"
          else
            # Include 2 decimal places if the duration is under a minute
            "#{secs.round(2)} sec"
          end
        end
      end
    end
  end
end
