module PuppetX
  module Util
    class Orchestrator

      # Methods for retrieving and processing /orchestrator/v1/job/#/report
      # specifically for logging and displaying the results of Puppet runs.
      module Report
        # Used by save_report_log to fetch the job report from the orchestration service
        # A valid response will be JSON-parsable, have a 'report' key with an array of reports
        # (which should really just be one report), and the 'events' key in this report should
        # either be nil (nothing changed during the puppet run), or a non-empty array of events
        # that happened during the run.
        #
        # Sometimes you can get malformed objects back if services on either the
        # remote or primary node have a hiccup, like getting an empty events array
        # or a bad response code. This function retries fetching the report for
        # up to 10 seconds when this happens, in case the error can correct
        # itself.
        #
        # PE-28018: Figure out which of these error symptoms are not actually
        # correctable and don't retry in those cases.
        def get_job_report(job_url)
          url = "#{job_url}/report"
          result = nil

          # It can take a second for the report to actually show up once
          # the job finishes.
          retry_time = 10
          start_time = Time.now
          while Time.now - start_time < retry_time
            uri = URI(url)
            response = orch_connection.get(uri, headers: orch_request_headers)
            begin
              check_orch_response(response, 200)
              report = JSON.parse(response.body)['report'].first()
              # If the events array is empty, something bad probably happened. If no changes were made,
              # events should be nil, not an empty array.
              if report['events']
                if report['events'].empty?
                  Puppet.debug(_('Report returned from orchestration service contained an empty events array'))
                  break
                end
                report['events'] = report['events'].sort_by { |s| Time.parse(s['timestamp']) }
              end
              result = report
              break
            rescue PuppetX::Util::OrchestratorResponseError => e
              Puppet.debug(_('Bad response from orchestration service when fetching report'))
              Puppet.debug(_("Status code: %{code}" % { code: e.status_code }))
              Puppet.debug(_("Response body: %{body}" % { body: e.body }))
            rescue PuppetX::Util::OrchestratorEmptyResponseError
              Puppet.debug(_('Empty body in response from orchestration service when fetching report'))
            rescue StandardError => e
              Puppet.debug(_("Error parsing response from orchestrator with the following message: %{msg}") % { msg: e.message })
              break
            end
            Puppet.debug(_("Retrying fetching of job report at %{url}" % { url: url }))
            sleep(orch_status_wait_seconds)
          end
          Puppet.debug(_('Could not fetch a valid response from orchestration service')) if result.nil?
          result || {}
        end

        # Expects a report as returned by get_job_report, or a Puppet::Transaction::Report object
        # This will extract all failed event types from the report and return them in an array.
        def find_errors(report)
          errors = []
          begin
            if report.is_a?(Puppet::Transaction::Report)
              report.resource_statuses.each do |res, status|
                if status.failed || status.failed_to_restart
                  message = ''
                  status.events.each do |thisevent|
                    message = message + thisevent.message + "\n" if thisevent.status =~ /fail/
                  end
                  event = {
                    'file'           => status.file,
                    'line'           => status.line,
                    'resource_type'  => status.resource_type,
                    'resource_title' => status.title,
                    'message'        => message,
                  }
                  errors << event
                end
              end
            else
              if report['events']
                report['events'].each do |event|
                  errors << event if event['status'] && event['status'] =~ /failure/
                end
              end
            end
          rescue StandardError => e
            Puppet.debug(_("Error parsing report to find errors with the following message: %{msg}") % { msg: e.message })
          end
          errors
        end

        # Expects a report as returned by get_job_report, or a Puppet::Transaction::Report object
        # Calls find_errors from within this function for convenience of not needing to call
        # both functions when you want to print errors.
        def print_errors(report)
          if report.is_a?(Puppet::Transaction::Report)
            return if report.status != 'failed'
          else
            return if report.empty? || (report['events'] && report['events'].empty?)
          end
          errors = find_errors(report)
          return if errors.empty?
          puts _('*** Run Errors ***')
          errors.each do |error|
            puts <<~EOS
              File:      #{error['file']}
              Line:      #{error['line']}
              Class:     #{error['class']}
              Resource:  #{error['resource_type'].capitalize}[#{error['resource_title']}]
              Message:   #{error['message']}

            EOS
          end
        end

        # Attempts to fetch the job run report from the orchestration service. If
        # it can, it writes it to the given log file and prints any errors from
        # that report.
        #
        # If it can't fetch a valid report from the orchestration service:
        #   If the puppet run was on the local node (primary)
        #     It attempts to load the last_run_report.yaml file out of cache. If
        #     that works, it saves it and prints the errors found.
        #     If the last_run_report is for a different job ID, then the puppet
        #     run probably never happened.
        #   If the puppet run was on a remote node
        #     We don't yet have the capability of reaching out to that node and
        #     fetching the last_run_report.yaml file, so we tell the user to do
        #     it if they want to.
        def save_report_log(display_scope, job, report_log)
          last_run_report = '/opt/puppetlabs/puppet/cache/state/last_run_report.yaml'
          begin
            report = get_job_report(job.url)
            if report.empty?
              Puppet.warning(_("Could not fetch the run report from the orchestration service."))
              if service_config[:server] =~ /#{display_scope}/
                last_run = YAML.safe_load(File.read(last_run_report), [Puppet::Transaction::Report])
                if last_run.job_id.to_s == job.id.to_s
                  FileUtils.copy(last_run_report, report_log)
                  puts _('Saving the local run report from this host instead.')
                  Puppet.debug(_('Fetched the last_run_report.yaml file from cache on this host as the report.'))
                  print_errors(last_run)
                else
                  Puppet.warning(_("We couldn't find a run report that matches this job ID, so the Puppet run likely never occurred. Try verifiying that PE services are running."))
                  return
                end
              else
                puts _("More information about this Puppet run (job ID %{job_id}) might be available in the run report stored on %{display_scope} at %{last_run_report}." % { job_id: job.id, last_run_report: last_run_report, display_scope: display_scope })
                return
              end
            else
              File.write(report_log, report.to_yaml)
              print_errors(report)
            end
            puts _("Puppet run report saved to %{report_log}" % { report_log: report_log })
          rescue StandardError => e
            Puppet.warning(_("Error saving report log with the following message: %{msg}") % { msg: e.message })
          end
        end
      end
    end
  end
end
