require 'json'
require 'net/https'
require 'puppet'

module PuppetX
  module Util
    class PasswordUtils

      RBAC_HOSTNAME = Puppet[:certname]
      SERVER_DATA_DIR = '/opt/puppetlabs/server/data'
      CONSOLE_CERT_DIR = "#{SERVER_DATA_DIR}/console-services/certs"

      CONF = {
        'rbac_api_url' => "https://#{RBAC_HOSTNAME}:4433",
        'ca_cert_file' => '/etc/puppetlabs/puppet/ssl/certs/ca.pem',
        'host_private_key_file' => "#{CONSOLE_CERT_DIR}/#{RBAC_HOSTNAME}.private_key.pem",
        'host_cert_file' => "#{CONSOLE_CERT_DIR}/#{RBAC_HOSTNAME}.cert.pem",
        'rbac_url_prefix' => '/rbac-api/v1',
      }

      def self.console_reset_admin_url
        user = generate_userhash('admin')
        "https://#{RBAC_HOSTNAME}/auth/reset?token=#{reset_token(user)}"
      end

      def self.reset_token(user)
        reset_token_res = rbac_request('post', "/users/#{user['id']}/password/reset")
        reset_token_res.body
      end

      def self.build_auth(uri)
        https = Net::HTTP.new(uri.host, uri.port, nil)
        https.use_ssl = true
        https.ca_file = CONF['ca_cert_file']
        https.key = OpenSSL::PKey::RSA.new(File.read(CONF['host_private_key_file']))
        https.cert = OpenSSL::X509::Certificate.new(File.read(CONF['host_cert_file']))
        https.verify_mode = OpenSSL::SSL::VERIFY_PEER
        https
      end

      def self.rbac_request(request_type, endpoint, request_body=nil, auth_failure_expected=false)
        uri = URI.parse("#{CONF['rbac_api_url']}#{CONF['rbac_url_prefix']}#{endpoint}")
        https = self.build_auth(uri)

        request = nil 
        case request_type
        when 'get'
          request = Net::HTTP::Get.new(uri.request_uri)
        when 'post'
          request = Net::HTTP::Post.new(uri.request_uri)
        when 'put'
          request = Net::HTTP::Put.new(uri.request_uri)
        end

        request['Content-Type'] = "application/json"
        unless request_body.nil?
          request.body = request_body.to_json
        end

        res = https.request(request)
        case res.code
        #Net:HTTPOK and Net::HTTPCreated
        when '200', '201' then
          res
        when '401' then
          if auth_failure_expected 
            res
          else
            tryparse_error(res, endpoint)
          end
        else
          tryparse_error(res, endpoint)
        end
      end

      def self.tryparse_error(res, endpoint)
        begin
          res_body = JSON.parse(res.body)
          error_body = "#{res_body['msg']} #{res_body['details']['reasons'].first}"
        rescue => exception
          error_body = res.body || ''
        end
        raise _("An error occured getting a response from RBAC while querying endpoint %{endpoint}: HTTP %{rescode}. %{error_body}") % {endpoint: endpoint, rescode: res.code, error_body: error_body}
      end

      def self.generate_userhash(user)
        userhash = self.get_user(user)
        if userhash.nil?
          raise _("User %{user} does not exist") % {user: user}
        end
        user = userhash
      end

      def self.get_user(user_name)
        user_list_res = self.rbac_request('get', '/users')
        user_list = JSON.parse(user_list_res.body)
        user = user_list.find { |user| user['login'] == user_name }
        user
      end

      def self.reset_password(user, new_password)
        user = generate_userhash(user) if user.class.name == 'String'
        reset_token = reset_token(user)

        reset_body = {
          'token' => reset_token,
          'password' => new_password,
        }
        self.rbac_request('post', '/auth/reset', reset_body)
      end

      def self.revoke_admin_access
        user_json = generate_userhash('admin')
        user_json['is_revoked'] = true

        self.rbac_request('put',"/users/#{user_json['id']}", user_json)
      end

      def self.check_for_default_password_and_revoke
        revoked = false

        user = generate_userhash('admin')
        token_request_body = { 
          'login' => user['login'],
          'password' => 'admin',
        }

        response = self.rbac_request('post', '/auth/token', token_request_body, true)
        if response.code != '401'
          self.revoke_admin_access
          revoked = true
        end
        revoked
      end
    end
  end
end
