require 'puppet'
require 'puppet/http'
require 'puppet/util/colors'
require 'puppet_x/util/postgresql_status'
require 'puppet_x/util/stringformatter'
require 'puppet_x/util/utilities'
require 'hocon'
require 'json'

module PuppetX
  module Util
    # Generic error raised by routines in this library.
    class ServiceStatusError < StandardError; end

    class ServiceStatus
      extend Puppet::Util::Colors
      extend PuppetX::Util::Utilities

      def self.config_path()
        "/etc/puppetlabs/client-tools/services.conf"
      end

      # Utility method to filter out services that aren't required on the replica from the list
      def self.replica_services()
        services = required_services
        [:orchestrator, :code_manager, :file_sync_storage, :pcp_broker].each do |svc|
          services.delete(svc)
        end
        services
      end

      def self.primary_services()
        required_services
      end

      # Using a dynamic version means these links may need occasional updates to avoid breakage.
      def self.pe_version()
        if ver = Facter.value('pe_server_version')
          ver.split('.')[0..1].join('.')
        else
          'latest'
        end
      end

      # Docs links below indicate something that may not be enabled out of the box.
      # Services without docs links are core to PE and cannot be disabled.
      def self.required_services()
        version = self.pe_version()
        {
          :classifier => {
            :type => 'classifier',
          },
          :master => {
            :type => 'master',
          },
          :orchestrator => {
            :type => 'orchestrator',
          },
          :pcp_broker => {
            :type => 'pcp-broker',
          },
          :code_manager => {
            :type => 'code-manager',
          },
          :file_sync_storage => {
            :type => 'file-sync-storage',
          },
          :file_sync_client => {
            :type => 'file-sync-client',
          },
        }
      end

      ### service status functions

      # A 'status hash' looks like this:
      # { :service => 'service name',
      #   :state => :error, :unknown, :unreachable, :starting, :stopping, or :running,
      #   :status => status information,
      #   :server => some-server.infra.net,
      #   :type "master",
      #   :url => https://some-server.ifra.net:1234/some-service
      #   :alerts => [{:severity => :error, :warning, or :info,
      #               :message => "help!"}]}

      # A service config hash is read from services.conf, and looks like this:
      # { :node_certname : "mom.dev",
      #   :port 8140,
      #   :prefix "",
      #   :primary true,
      #   :server "mom.dev",
      #   :status_prefix "status",
      #   :status_url "https://mom.dev:8140/status",
      #   :type "master",
      #   :url "https://mom.dev:8140/",
      #   :status_key 'key-in-status-response-map'

      def self.get_service_on_primary(service)
        config = PuppetX::Util::ServiceStatus.load_services_config
        nodes_config = PuppetX::Util::ServiceStatus.load_nodes_config
        primary_master_node = nodes_config.find { |node| node[:role] == 'primary_master' }
        config.find { |svc| svc[:type] == service && svc[:node_certname] == primary_master_node[:certname] }
      end

      def self.reachable_status(svc, state, status_details, alerts)
        {
          :service => svc[:status_key],
          :state => state.to_sym,
          :status => status_details,
          :display_name => svc[:display_name],
          :server => svc[:server],
          :url => svc[:url],
          :status_url => svc[:status_url],
          :type => svc[:type],
          :alerts => alerts,
        }
      end

      # A bolt/ace-server Puma server is considered running if we received a
      # response. The details of the response reflect the state of the thread pool.
      #
      # Example:
      #
      #   {
      #     "started_at"=>"2020-04-21T17:06:11Z",
      #     "backlog"=>0,
      #     "running"=>1,
      #     "pool_capacity"=>9,
      #     "max_threads"=>10
      #   }
      #
      # Ref:
      # https://github.com/puppetlabs/bolt/blob/2.6.0/lib/bolt_server/transport_app.rb#L238-L241
      # https://github.com/puma/puma/blob/4.3.1/lib/puma.rb#L18-L24
      # https://github.com/puma/puma/blob/4.3.1/lib/puma/launcher.rb#L87-L90
      # https://github.com/puma/puma/blob/4.3.1/lib/puma/single.rb#L16-L22
      #
      # @return [Hash] a {reachable_status()} response.
      def self.translate_puma_status(svc, status)
        reachable_status(svc, :running, status, [])
      end

      # @return [Hash] either a {reachable_status()} or an {unreachable_status()}
      # hash based on whether the response_hash (which has responses for multiple
      # tk services) has an entry for +svc+.
      def self.translate_trapper_keeper_status(svc, response_hash)
        status = response_hash[svc[:status_key]]

        if status.nil?
          unreachable_status_for_service(svc)
        else
          alerts = (status['active_alerts'] || []).map do |a|
            {
              :severity => a['severity'].to_sym,
              :message => a['message']
            }
          end

          reachable_status(svc, status['state'], status['status'], alerts)
        end
      end

      # Call the status api at status_service_url and return an array of status hashes.
      # Cross-check with expected_service_names to determine which services
      # were unreachable.
      def self.try_get_status(status_service_url, expected_service_configs, timeout_seconds)
        begin
          uri = URI.parse(status_service_url)
          server = uri.host
          port = uri.port
          uri_with_timeout = URI("#{status_service_url}?timeout=#{timeout_seconds}")
          if Puppet.runtime.instance_variable_get(:@runtime_services).keys.include? :http
            runtime_service = :http
          else
            runtime_service = 'http'
          end
          connection = Puppet.runtime[runtime_service]
          options = { 'open_timeout' => timeout_seconds, 'read_timeout' => timeout_seconds }
          headers = {'Content-Type' => 'application/json'}
          # The timeout parameter is internal to tk service checks, is ignored by puma.
          # The timeout options are for the connection attempt itself.
          response = connection.get(uri_with_timeout, headers: headers, options: options)
          response_hash = JSON.parse(response.body)

          expected_service_configs.map do |svc|
            case svc[:type]
            when 'bolt', 'ace'
              translate_puma_status(svc, response_hash)
            else
              translate_trapper_keeper_status(svc, response_hash)
            end
          end
            # It would be awesome to be more explicit here but I'm not positive
            # what exceptions HttpPool raises. Maybe we can add exceptions until
            # we have an exhaustive list.
        rescue JSON::ParserError, Exception => e
          Puppet.debug _("Exception raised while trying to fetch status for %{url}") % { url: status_service_url }
          Puppet.debug "#{e.class}: #{e.message}"
          expected_service_configs.map { |svc| unreachable_status_for_service(svc)}
        end
      end

      def self.services_for(host, service, config_services, timeout_seconds, nodes_config:)
        host_services = config_services.select { |svc| host ? svc[:server] == host : true }

        matched_services = host_services.select do |svc|
          service ? svc[:type].downcase == service.downcase : true
        end

        master = get_master(nodes_config)
        available_services = matched_services.select do |svc|
          ['bolt', 'ace'].include?(svc[:type]) ?
            (master == Puppet[:certname]) : # bolt/ace only reachable on the primary
            true # other services can be reached from an allowlisted node (like the replica)
        end

        grouped_services = available_services.group_by { |svc| svc[:status_url] }
        grouped_services.flat_map do |status_service_url, configured_services|
          case status_service_url
          when %r{^postgresql:}
            PuppetX::Util::PostgresqlStatus.try_get_postgresql_status(
              status_service_url,
              configured_services,
              timeout_seconds,
              nodes_config: nodes_config,
              master: master,
              replicas: get_replicas(nodes_config),
            )
          else
            try_get_status(status_service_url, configured_services, timeout_seconds)
          end
        end
      end

      def self.node_role(nodes_config, certname)
        return nil unless nodes_config

        node_record = nodes_config.find { |node| node[:certname] == certname }
        node_record[:role] if node_record
      end

      def self.load_config(path_override)
        path = path_override || self.config_path()
        unless File.file?(path)
          message = [
            _("No configuration file found at %{path}.") % { path: path },
            _("This file is installed automatically on Puppet Server nodes."),
            _("Make sure you are running the command on a primary, replica, or compiler."),
          ]
          raise(PuppetX::Util::ServiceStatusError, PuppetX::Util::String::Formatter.wrap_no_indent(message.join(' '))) # rubocop:disable GetText/DecorateFunctionMessage
        end
        config = Hocon.load(path)
        symbolize_keys(config)
      end

      def self.load_services_config(path_override=nil)
        self.load_config(path_override)[:services]
      end

      def self.load_nodes_config(path_override=nil)
        self.load_config(path_override)[:nodes]
      end

      # @return [Array<String>] list of all replica certnames from services.conf.
      def self.get_replicas(nodes_config = nil)
        nodes_config ||= PuppetX::Util::ServiceStatus.load_nodes_config
        replica_nodes = nodes_config.select { |node| node[:role] == 'primary_master_replica' }
        replica_nodes.map{ |node| node[:certname] }
      end

      # @return [String] certname of primary. Assumes services.conf nodes only lists
      # a single primary.
      def self.get_master(nodes_config = nil)
        nodes_config ||= PuppetX::Util::ServiceStatus.load_nodes_config
        master_node = nodes_config.select { |node| node[:role] == 'primary_master' }
        master_node.map{ |node| node[:certname] }.first
      end

      # Given the parsed services array from service.conf, select the services
      # that are present on the given +certname+ and check them against the
      # given list of +required_services+.
      #
      # The intention is to provide a services hash in the correct format for
      # other HA methods to consume (a key, value map of :service => service_hash),
      # along with a list of missing services and a list of error messages about
      # the missing services.
      #
      # @param config [Array<Hash>] output of self.load_services_config()
      # @param certname [String] the certname to match services against (this
      # will likely be either the primary or the replica, or a mistake...)
      # @param required_services [Array<Hash>] required services to validate;
      # self.required_services() for example...
      # @return [Array] Returns three elements:
      # - service_hash, keyed by service symbol, the value is the Hash of service
      #   info parsed from services.conf. This is the Hash other HA methods will
      #   use to connect to a particular service.
      # - failed_services, Array of service names that were not found.
      # - errors, Array of error messages about these missing services (may have
      #   docs links.
      def self.validate_required_services(config, certname, required_services)
        errors = []
        failed_services = []
        service_hash = required_services.each_with_object({}) do |required, hash|
          required_svc_name = required.first
          required_svc_hash = required.last
          cur_config = config.find do |cur_service|
            cur_service[:node_certname] == certname &&
              cur_service[:type] == required_svc_hash[:type]
          end
          if cur_config
            hash[required_svc_name] = cur_config
          else
            # Code Manager should be capitalized and file sync should be
            # lower case to conform with docs. It's more clear for the user
            # to group these together, since they are all enabled together.
            case required_svc_hash[:type]
            when 'code-manager'
              svc = 'Code Manager'
            when 'file-sync-storage'
              svc = 'file sync'
            when 'file-sync-client'
              svc = 'file sync'
            else
              svc = required_svc_hash[:type]
            end
            failed_services << svc unless failed_services.include?(svc)
            unless ['Code Manager','file sync'].include?(svc)
              errors << _("Missing %{type} service.") % { type: required_svc_hash[:type] } + " " + _("This probably means this command hasn't been run on the primary node.")
            end
          end
        end

        # (PE-29819) We removed old docs links for code-manager, file-sync-storage, and file-sync-client,
        # as these three are now enabled together and we want users to go look at the latest docs.
        if failed_services.any?(/Code Manager|file sync/)
          errors << _("For instructions on enabling Code Manager (and file sync, which is automatically enabled with Code Manager), see the PE documentation.")
        end
        return service_hash, failed_services, errors
      end

      # Filters the config hash for only services defined on the node the command
      # was run on (assumption being that no commands need to directly contact any
      # services not on the current host), checks that the config hash has the
      # entries for all key services (defined in the SERVICES constant), pulls them
      # out, and returns them in a map.
      def self.validate_and_select_services_from_config(config, action, certname, services = self.required_services())
        service_hash, failed_services, errors = validate_required_services(config, certname, services)

        if errors.empty?
          service_hash
        else
          # Ideally this message would be handled outside of this validation
          # functions but as a temporary measure to prevent an erroroneous
          # message from being printed we check which action we're dealing with
          # to print the correct message
          command_string = action == 'promote' ? "#{action} replica" : "#{action} replica HOSTNAME"
          if failed_services.count == 1
            failedstr = "#{failed_services.first} is"
          else
            failedstr = "#{failed_services[0..-2].join(', ')} and #{failed_services[-1]} are"
          end
          error_string = _("Puppet cannot %{action} a replica until %{failedstr} enabled.") % { action: action, failedstr: failedstr }
          error_string << "\n" << errors.join(" ") << "\n"
          error_string << _("Then, run 'puppet infrastructure %{command_string}' to %{action} a replica.") % { command_string: command_string, action: action }
          raise(PuppetX::Util::ServiceStatusError, error_string) # rubocop:disable GetText/DecorateFunctionMessage
        end
      end

      def self.services_running(status)
        status.select {|service| service[:state] == :running}
      end

      def self.services_not_running(status)
        status.select {|service| service[:state] != :running}
      end

      def self.all_services_running?(status)
        self.services_running(status).count == status.count
      end

      def self.ensure_all_services_running(status)
        if self.all_services_running?(status)
          return true
        else
          errors = []
          self.services_not_running(status).each do |service|
            errors << _("%{display_name} is not running") % { display_name: service[:display_name] }
          end

          error_string = errors.join("\n")
          error_string << "\n" + _("Puppet cannot provision a replica until all required services are running.")
          raise(PuppetX::Util::ServiceStatusError, error_string) # rubocop:disable GetText/DecorateFunctionMessage
        end
      end

      ### Status formatting

      def self.service_has_changing_state(status_hash)
        [:starting, :stopping].include? status_hash[:state]
      end

      def self.service_has_error_state(status_hash)
        [:error, :unknown, :unreachable].include? status_hash[:state]
      end

      def self.service_has_important_alert(status_hash)
        alert_severities = status_hash[:alerts].map { |a| a[:severity] }
        (alert_severities & [:error, :warning]).count > 0  # & is set intersection
      end

      def self.format_server_status_header(server_display_name, server)
         "\n#{server_display_name}: #{server}\n"
      end

      def self.format_server_status_footer
         "\n"
      end

      def self.format_summary_message(service_statuses_count, error_service_statuses_count)
        operational_service_count = service_statuses_count - error_service_statuses_count
        color = error_service_statuses_count == 0 ? :green : :yellow
        colorized_summary = colorize(color, _("%{operational_service_count} of %{service_statuses_count} services are fully operational.") %
          { operational_service_count: operational_service_count, service_statuses_count: service_statuses_count })
        "Status at #{Time.now}\n#{colorized_summary}\n"
      end

      def self.format_status(status)
        alerts_text = status[:alerts]
                          .map { |alert| "    #{alert[:severity].to_s.capitalize}: #{alert[:message]}\n" }
                          .join("")
        colored_summary = "#{status[:display_name]}: #{status[:state].to_s.capitalize}"
        if service_has_error_state(status)
          colored_summary = colorize(:red, colored_summary)
        elsif service_has_changing_state(status) || service_has_important_alert(status)
          colored_summary = colorize(:yellow, colored_summary)
        elsif :running == status[:state]
          colored_summary = colorize(:green, colored_summary)
        end
        "  #{colored_summary}, checked via #{status[:status_url]}\n#{alerts_text}"
      end
    end
  end
end
