# Ciphers before TLS 1.3 have an IANA/JSSE format, and an OpenSSL format.
# Jetty/Java requires the former, orch/bolt-server/ace-server/nginx/postgres the latter.
# Translations between the two can be found at
# https://testssl.sh/openssl-iana.mapping.html
# https://www.openssl.org/docs/man1.1.1/man1/ciphers.html
#
# Cipher lists are derived from https://confluence.puppetlabs.com/display/SRE/TLS+Usage+Standard
# which is derived from the Mozilla Intermediate list.
#
# -----------------------------------------------------------------------------
#  IMPORTANT DETAILS FOR USAGE IN PUPPET ENTERPRISE
#  This function can be used as the authoritative list of cipher defaults used
#  in Puppet Enterprise with the following caveats.
# -----------------------------------------------------------------------------
# - Because puppet/puppetserver only generates RSA keys, we effectively only use the RSA ciphers here. 
#   We might explore ECC keys to support ECDSA in the future.  However, we include a couple ECDSA ciphers 
#   in case a user wants to generate their own certs/keys. Be aware, though, that these suites won't be useable
#   by a regular PE install or in the browser with the PE console using puppet/puppetserver-generated certs/keys.
# - FIPS_allowed is the list of ciphers allowed on FIPS installs. This is dervied by collecting all the ciphers we 
#   use in PE, and then filtering based on https://www.gsa.gov/cdnstatic/SSL_TLS_Implementation_%5BCIO_IT_Security_14-69_Rev_4%5D_5-26-2020docx.pdf
# - TLSv1 and TLSv1.1 lists are the same. Copied here so one or both can be passed in to the function.
# - 'browser_' ciphers are used for Nginx. Currently the same as the other lists + CHACHA ciphers, but listed verbosely here for clarity.
# - The following ciphers are omitted (except from the browser list) vs. the Confluence page list due to Java 11 not supporting it.
#   https://github.com/AdoptOpenJDK/openjdk-jdk11/blob/19fb8f93c59dfd791f62d41f332db9e306bc1422/src/java.base/share/classes/sun/security/ssl/CipherSuite.java#L483
#   TLS_CHACHA20_POLY1305_SHA256
#   TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 (ECDHE-ECDSA-CHACHA20-POLY1305)
#   TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256 (ECDHE-RSA-CHACHA20-POLY1305)
#   These will be added in Java 12. https://bugs.java.com/bugdatabase/view_bug.do?bug_id=8210799
# - Though it does not appear here, Puma-based services (bolt-server/ace-server) will also allow
#   TLS_CHACHA20_POLY1305_SHA256 for TLS 1.3 in non-FIPS installs. This is because Puma uses SSL_CTX_set_cipher_list()
#   to set the allowed cipher lists, but this only works with 1.2 ciphers, and 1.3 ciphers
#   require the use of SSL_CTX_set_ciphersuites()   *facepalm*
#   https://www.openssl.org/docs/man1.1.1/man3/SSL_CTX_set_ciphersuites.html
# - Puma-based services are unable to limit the SSL protocol used, so regardless of the cipher
#   list provided or the puppet_enterprise::ssl_protocols setting, it will allow all TLS versions.
#   This may change in the future to disallow TLSv1 and TLSv1.1.
# - PostgreSQL 11 does not limit the SSL protocols either, but once we move to 14, it will. However, there is
#   no setting in postgresql.conf for controlling 1.3 ciphers, only 1.2 or less.

function puppet_enterprise::ciphers(
  Array[Enum['TLSv1.3', 'TLSv1.2', 'TLSv1.1', 'TLSv1', 'browser_TLSv1.3', 'browser_TLSv1.2', 'browser_TLSv1.1', 'browser_TLSv1']] $cipher_groups,
  Enum['openssl', 'iana'] $format,
) >> Array[String] {
  # When updating these lists, ensure both the openssl
  # and iana formats are updated. Note that 'browser' only
  # shows up in the openssl format since it is only used
  # for nginx.
  $ciphers = {
    'openssl' => {
      'TLSv1.3' => [
        'TLS_AES_256_GCM_SHA384',
        'TLS_AES_128_GCM_SHA256',
        # Puma-based services (bolt/ace server) will add TLS_CHACHA20_POLY1305_SHA256 to this list
      ],
      'TLSv1.2' => [
        'ECDHE-RSA-AES256-GCM-SHA384',
        'ECDHE-ECDSA-AES256-GCM-SHA384',
        'ECDHE-RSA-AES128-GCM-SHA256',
        'ECDHE-ECDSA-AES128-GCM-SHA256',
        'DHE-RSA-AES256-GCM-SHA384',
        'DHE-RSA-AES128-GCM-SHA256',
      ],
      'TLSv1.1' => [
        'ECDHE-RSA-AES256-SHA',
        'ECDHE-ECDSA-AES256-SHA',
        'ECDHE-RSA-AES128-SHA',
        'ECDHE-ECDSA-AES128-SHA',
        'DHE-RSA-AES256-SHA',
        'DHE-RSA-AES128-SHA',
      ],
      'TLSv1' => [
        'ECDHE-RSA-AES256-SHA',
        'ECDHE-ECDSA-AES256-SHA',
        'ECDHE-RSA-AES128-SHA',
        'ECDHE-ECDSA-AES128-SHA',
        'DHE-RSA-AES256-SHA',
        'DHE-RSA-AES128-SHA',
      ],
      'browser_TLSv1.3' => [
        'TLS_AES_256_GCM_SHA384',
        'TLS_AES_128_GCM_SHA256',
        'TLS_CHACHA20_POLY1305_SHA256',
      ],
      'browser_TLSv1.2' => [
        'ECDHE-RSA-AES256-GCM-SHA384',
        'ECDHE-ECDSA-AES256-GCM-SHA384',
        'ECDHE-RSA-AES128-GCM-SHA256',
        'ECDHE-ECDSA-AES128-GCM-SHA256',
        'ECDHE-RSA-CHACHA20-POLY1305',
        'ECDHE-ECDSA-CHACHA20-POLY1305',
        'DHE-RSA-AES256-GCM-SHA384',
        'DHE-RSA-AES128-GCM-SHA256',
      ],
      'browser_TLSv1.1' => [
        'ECDHE-RSA-AES256-SHA',
        'ECDHE-ECDSA-AES256-SHA',
        'ECDHE-RSA-AES128-SHA',
        'ECDHE-ECDSA-AES128-SHA',
        'DHE-RSA-AES256-SHA',
        'DHE-RSA-AES128-SHA',
      ],
      'browser_TLSv1' => [
        'ECDHE-RSA-AES256-SHA',
        'ECDHE-ECDSA-AES256-SHA',
        'ECDHE-RSA-AES128-SHA',
        'ECDHE-ECDSA-AES128-SHA',
        'DHE-RSA-AES256-SHA',
        'DHE-RSA-AES128-SHA',
      ],
      'FIPS_allowed' => [
        'TLS_AES_256_GCM_SHA384',
        'TLS_AES_128_GCM_SHA256',
        'ECDHE-RSA-AES256-GCM-SHA384',
        'ECDHE-ECDSA-AES256-GCM-SHA384',
        'ECDHE-RSA-AES128-GCM-SHA256',
        'ECDHE-ECDSA-AES128-GCM-SHA256',
        'ECDHE-RSA-AES256-SHA',
        'ECDHE-ECDSA-AES256-SHA',
        'ECDHE-RSA-AES128-SHA',
        'ECDHE-ECDSA-AES128-SHA',
      ],
    },
    'iana' => {
      'TLSv1.3' => [
        'TLS_AES_256_GCM_SHA384',
        'TLS_AES_128_GCM_SHA256',
        # Puma-based services (bolt/ace server) will add TLS_CHACHA20_POLY1305_SHA256 to this list
      ],
      'TLSv1.2' => [
        'TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384',
        'TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384',
        'TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256',
        'TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256',
        'TLS_DHE_RSA_WITH_AES_256_GCM_SHA384',
        'TLS_DHE_RSA_WITH_AES_128_GCM_SHA256',
      ],
      'TLSv1.1' => [
        'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA',
        'TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA',
        'TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA',
        'TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA',
        'TLS_DHE_RSA_WITH_AES_256_CBC_SHA',
        'TLS_DHE_RSA_WITH_AES_128_CBC_SHA',
      ],
      'TLSv1' => [
        'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA',
        'TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA',
        'TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA',
        'TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA',
        'TLS_DHE_RSA_WITH_AES_256_CBC_SHA',
        'TLS_DHE_RSA_WITH_AES_128_CBC_SHA',
      ],
      'FIPS_allowed' => [
        'TLS_AES_256_GCM_SHA384',
        'TLS_AES_128_GCM_SHA256',
        'TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384',
        'TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384',
        'TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256',
        'TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256',
        'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA',
        'TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA',
        'TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA',
        'TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA',
      ],
    }
  }

  $requested_ciphers = $cipher_groups.map |$cipher_group| { $ciphers[$format][$cipher_group] }.flatten.unique

  # Probably shouldn't be able to pass in an empty protocol list, 
  # but just in case.
  if empty($requested_ciphers) {
    $cipher_list = $ciphers[$format]['TLSv1.3'] + $ciphers[$format]['TLSv1.2']
  } else {
    $cipher_list = $requested_ciphers
  }

  if $facts['fips_enabled'] {
    $cipher_list.filter |$cipher| { $cipher in $ciphers[$format]['FIPS_allowed'] }
  } else {
    $cipher_list
  }
}