# This class manages pe-installer's bolt.yaml file
#
# @param bolt_yaml_file [String] Path to the installer's bolt.yaml file
# @param bolt_modulepath [String] Bolt modulepath to set in bolt-project.yaml. This should include the installer's site-modules folder as well as the PE modules path
# @param bolt_log_file [String] Path to log file to save the bolt run log
# @param bolt_log_level [String] Bolt logging level
# @param certname [String] Certname of the node where this manifest is being applied. Used to determine if the node is the primary or replica.
# @param puppetdb_hosts Array[String] Array of certnames of nodes running the PuppetDB service that Bolt should talk to for PDB queries.
# @param puppetdb_ports Array[Integer] The ports the PuppetDB services are listening on. May be an array size of 1 if all hosts use the same port (they should).
# @param cacert [String] Path to the local certificate authority pem file
# @param cert [String] Path to the node's certificate pem file
# @param key [String] Path to the node's private key pem file
class puppet_enterprise::master::installer_bolt(
  String $bolt_yaml_file          = '/opt/puppetlabs/installer/share/Boltdir/bolt-project.yaml',
  String $bolt_modulepath         = '/opt/puppetlabs/installer/share/Boltdir/site-modules:/opt/puppetlabs/server/data/environments/enterprise/modules:/opt/puppetlabs/installer/share/Boltdir/modules',
  String $bolt_log_file           = '/var/log/puppetlabs/installer/bolt_info.log',
  String $bolt_log_level          = 'info',
  String $certname                = $facts['clientcert'],
  Array[String] $puppetdb_hosts   = $puppet_enterprise::puppetdb_hosts_array,
  Array[Integer] $puppetdb_ports  = $puppet_enterprise::puppetdb_ports_array,
  String $cacert                  = $puppet_enterprise::params::localcacert,
  String $cert                    = "${puppet_enterprise::params::ssl_dir}/certs/${certname}.pem",
  String $key                     = "${puppet_enterprise::params::ssl_dir}/private_keys/${certname}.pem"
) {
  if($certname == $puppet_enterprise::puppet_master_host or $certname in $puppet_enterprise::ha_enabled_replicas) {
    pe_concat { $bolt_yaml_file:
      owner           => 'root',
      group           => 'root',
      mode            => '0644',
      warn            => true,
      replace         => true,
      ensure_newline  => true
    }

    pe_concat::fragment { 'bolt_yaml_header':
      target  => $bolt_yaml_file,
      content => '---',
      order   => '1',
    }

    pe_concat::fragment { 'bolt_yaml_modulepath': 
      target  => $bolt_yaml_file,
      content => "modulepath: \"${bolt_modulepath}\"",
      order   => '2',
    }

    pe_concat::fragment { 'bolt_yaml_log':
      target  => $bolt_yaml_file,
      content => "log:\n  ${bolt_log_file}:\n    level: ${bolt_log_level}",
      order   => '3',
    }

    # We only want to prepend $puppetdb_hosts with the node's certname on a replica because
    # 1) The replica is not included in puppetdb_hosts_array, so we need to add it if we're on the replica
    # 2) On a mono or mono+postgres install, the primary will be included in $puppetdb_hosts_array already
    # 3) On a legacy split, the primary is not the PuppetDB node so we need to NOT include it
    if ($certname in $puppet_enterprise::ha_enabled_replicas) {
      $pdb_hosts = pe_union([$certname],$puppetdb_hosts)
    } else {
      $pdb_hosts = $puppetdb_hosts
    }
    $pdb_server_urls = pe_prefix(pe_suffix(pe_format_urls('https', $pdb_hosts, [$puppetdb_ports[0]]), '"'), '"')
    pe_concat::fragment { 'bolt_yaml_puppetdb':
      target  => $bolt_yaml_file,
      content => "puppetdb:\n  server_urls: ${pdb_server_urls}\n  cacert: ${cacert}\n  cert: ${cert}\n  key: ${key}",
      order   => '4',
    }

    # Bolt 2.38 adds a spinner to the console for long running Bolt operations. Since we run Bolt via PTY, this
    # may screw up our capture of stdout.
    pe_concat::fragment { 'bolt_yaml_disable_spinner':
      target  => $bolt_yaml_file,
      content => "spinner: false",
      order   => '5',
    }
   file { '/opt/puppetlabs/installer/share/Boltdir/bolt.yaml':
     ensure => absent,
   }

  }
}
