# A define type to manage the creation and permission juggling of a user
# which coordinates postgresql connections during migrations. This migrator
# role has the ability to revoke connect from the regular non-migrator and
# is granted membership to the same role as the non-migrator. Membership in
# the non-migrator role allows the migrator to 'set role ...;' and become
# the non-migrator after it has killed all the other non-migrator connections.
# Doing this allows the migrator to ensure it is the only one connected to the
# database during a migration and ensure that it can create objects which are
# still owned by the non-migrator role.
#
# Using pe-puppetdb as an example this setup should result in the access
# privileges for the database being defined as shown in the example below:
#
# =T/"pe-postgres"
# "pe-postgres"=CTc/"pe-postgres"
# "pe-puppetdb"=CT/"pe-postgres"
# "pe-puppetdb-migrator"=c*/"pe-postgres"
# "pe-puppetdb"=c/"pe-puppetdb-migrator"
#
# @param non_migrator [String] The name of the non-migrator postgres user
# @param migrator [String] The name of the migrator postgres user
# @param database_name [String] The name of the database to grant access to.
# @param tablespace_name [String] The name of the tablespace for this database.
# @param password_hash [String] The value of $_database_password in app_database.

define puppet_enterprise::pg::migrator_user (
  String $non_migrator,
  String $migrator,
  String $database_name,
  String $tablespace_name,
  Variant[String,Boolean] $password_hash = false,
) {

  puppet_enterprise::psql { "REVOKING CONNECT ON DATABASE \"${database_name}\" FROM public":
    command => "REVOKE CONNECT ON DATABASE \"${database_name}\" FROM public",
    require => Pe_postgresql::Server::Db[$database_name],
    unless  => "SELECT * from has_database_privilege(
                 'public',
                 '${database_name}',
                 'CONNECT')
               WHERE has_database_privilege = false"
  }

  pe_postgresql::server::role { $migrator:
    password_hash => $password_hash,
    require       => Pe_postgresql::Server::Tablespace[$tablespace_name],
  } ->

  # Include the migrator user in the non_migrator role. This allows the migrator
  # to `set role non_migrator;` later on so any database objects it creates will
  # be owned by the non_migrator.
  puppet_enterprise::psql { "GRANT ${non_migrator} TO ${migrator}":
    command => "GRANT \"${non_migrator}\" TO \"${migrator}\"",
    require => Pe_postgresql::Server::Db[$database_name],
    unless => "SELECT * FROM pg_has_role(
                '${migrator}',
                '${non_migrator}',
                'member')
              WHERE pg_has_role = true"
  } ->

  # Revoke existing connect grant from pe-postgres so the migrator can control connect access.
  puppet_enterprise::psql { "${database_name} revoke pe-postgres's connect grant from ${non_migrator}":
    command => "REVOKE CONNECT ON DATABASE \"${database_name}\" FROM \"${non_migrator}\"",
    require => Pe_postgresql::Server::Db[$database_name],
    unless  => "SELECT * FROM pg_catalog.pg_database
                 WHERE datname = '${database_name}'
                 AND NOT datacl @> '\"${non_migrator}\"=c/\"pe-postgres\"'::aclitem"
  } ->

  # Grant connect to the migrator with the ability to grant connect to other roles.
  puppet_enterprise::psql { "GRANT ${migrator} - CONNECT - ${database_name} WITH GRANT OPTION":
    command => "GRANT CONNECT ON DATABASE \"${database_name}\" TO \"${migrator}\" WITH GRANT OPTION",
    require => Pe_postgresql::Server::Db[$database_name],
    unless  => "SELECT * FROM has_database_privilege(
                 '${migrator}',
                 '${database_name}',
                 'CONNECT WITH GRANT OPTION')
               WHERE has_database_privilege = true"
  } ->

  # Grant connect to the non_migrator from the migrator. This allows the migrator to control all
  # connections to the database during migrations.
  puppet_enterprise::psql { "SET ROLE ${migrator}; GRANT ${non_migrator} - CONNECT - ${database_name}":
    command => "SET ROLE \"${migrator}\"; GRANT CONNECT ON DATABASE \"${database_name}\" TO \"${non_migrator}\"",
    require => Pe_postgresql::Server::Db[$database_name],
    unless  => "SELECT * FROM pg_catalog.pg_database
                 WHERE datname = '${database_name}'
                 AND datacl @> '\"${non_migrator}\"=c/\"${migrator}\"'::aclitem"
  }
}
