define puppet_enterprise::pg::pglogical::subscription(
  String $subscription_name,
  String $database,
  String $host = '',
  String $user = '',
  String $ssl_cert_file = '',
  String $ssl_key_file = '',
  String $ssl_ca_file = '',
  Enum[present, absent] $ensure = present,
  Integer $keepalives_idle = $puppet_enterprise::pglogical_keepalives_idle,
  Optional[Integer] $keepalives_interval = $puppet_enterprise::pglogical_keepalives_interval,
  Integer $keepalives_count = $puppet_enterprise::pglogical_keepalives_count,
  Boolean $synchronize_structure = true,
  Boolean $enabled = true,
) {
  # lint:ignore:case_without_default
  case $ensure {
    present: {
      $provider_dsn_hash = {
        'host'                => $host,
        'port'                => $pe_postgresql::params::port,
        'dbname'              => $database,
        'user'                => $user,
        'sslmode'             => 'verify-full',
        'sslcert'             => $ssl_cert_file,
        'sslkey'              => $ssl_key_file,
        'sslrootcert'         => $ssl_ca_file,
        'keepalives_idle'     => $keepalives_idle,
        'keepalives_interval' => $keepalives_interval,
        'keepalives_count'    => $keepalives_count,
      }

      $provider_dsn = pe_hash2dsn($provider_dsn_hash)

      if $enabled {
        puppet_enterprise::psql { "pglogical_subscription ${title} create-sql":
          db      => $database,
          command => "SELECT pglogical.create_subscription(
                        subscription_name := '${subscription_name}',
                        provider_dsn := '${provider_dsn}',
                        synchronize_structure := ${synchronize_structure})",
          unless  => "SELECT * from pglogical.subscription WHERE sub_name='${subscription_name}'",
        }
        -> puppet_enterprise::psql { "pglogical_subscription ${title} enable-sql":
          db      => $database,
          command => "SELECT pglogical.alter_subscription_enable('${subscription_name}')",
          unless  => "SELECT * from pglogical.subscription WHERE sub_name='${subscription_name}' and sub_enabled = true",
        }
      } else {
        # The union on the unless clause ensures we don't try to disable when no
        # subscription exists yet.
        puppet_enterprise::psql { "pglogical_subscription ${title} disable-sql":
          db      => $database,
          command => "SELECT pglogical.alter_subscription_disable('${subscription_name}')",
          unless  => "SELECT sub_name from pglogical.subscription
                      WHERE sub_name='${subscription_name}' and sub_enabled = false
                      UNION
                      SELECT 'no-subscription'
                      WHERE not exists (SELECT * from pglogical.subscription
                                        WHERE sub_name = '${subscription_name}')",
        }
      }
    }

    absent: {
      puppet_enterprise::psql { "pglogical_subscription ${title} drop-sql":
        db      => $database,
        command => "SELECT pglogical.drop_subscription('${subscription_name}', false)",
        # query that returns rows if the subscription doesn't exist
        unless  => "SELECT *
                    FROM (SELECT COUNT(*)
                          FROM pglogical.subscription
                          WHERE sub_name='${subscription_name}') c
                    WHERE c.count = 0",
      }
    }
  }
  # lint:endignore
}
