# This class is for configuring a node to be a PuppetDB node for an environment.
#
# For more information, see the [README.md](./README.md)
#
# @param database_host The host which the database server is running on
# @param master_certname The primary's certificate name
# @param whitelisted_certnames * Deprecated * Use allowlisted_certnames instead.
# @param allowlisted_certnames An array of certificates allowed to communicate directly with
#        PuppetDB. This list is added to the base PE certificate list.
# @param certname The certname of the host that PuppetDB is running on
# @param confdir The path to PuppetDB's confdir
# @param database_name The name of the PuppetDB Database
# @param database_password The password of the user
# @param database_port The port that the database is running on
# @param database_user The user logging into the database
# @param gc_interval The interval, in minutes, at which garbage collection should occur
# @param node_purge_ttl The amount of time that must elapse before a deactivated node is
#        purged from PuppetDB
# @param node_ttl The amount of time that must elapse before a node is deactivated from
#        PuppetDB
# @param report_ttl The amount of time that must elapse before a report is deleted
# @param listen_address The address which the database is listening on for plain text
#        connections
# @param listen_port The port which PuppetDB Listens on for plain text connections
# @param ssl_listen_address The address which the database is listening on for SSL
#        connections
# @param ssl_listen_port The port which PuppetDB Listens on for SSL connections
# @param localcacert The path to the local CA certificate. This will be used instead of the
#        CA that is in Puppet's ssl dir.
# @param database_properties A url encoded string of JDBC options.  This will replace the
#        default database property string which enables SSL connections.
# @param java_args A hash containing Java options that puppetdb will run with. This will
#        replace any arguments that are used by default.
# @param auto_configure_sync This setting is deprecated1
# @param sync_peers The hosts from which this puppetdb should pull data, along
#        with the interval at which the sync should be run.
# @param sync_whitelist * Deprecated * Use sync_allowlist instead.
# @param sync_allowlist Certnames which are allowed to sync from this puppetdb.
# @param syncing [Boolean] Toggles on or off puppetdb synchronization (assuming
#        sync_peers and sync_allowlist have been set so that synchronization is
#        possible). Defaults to true. May be temporarily set false by PE
#        management routines to pause syncing.
# @param rbac_host The hostname of the RBAC node.
# @param rbac_port The port of the RBAC service.
# @param rbac_prefix The URL prefix of the RBAC service.
# @param enable_system_metrics_collection [Boolean] Enable/disable collection of system metrics.
class puppet_enterprise::profile::puppetdb(
  String                          $database_host          = $puppet_enterprise::puppetdb_database_host,
  String                          $master_certname        = $puppet_enterprise::puppet_master_host,
  Optional[Array[String]]         $whitelisted_certnames  = undef,
  Array[String]                   $allowlisted_certnames  = pe_pick($whitelisted_certnames, []),
  String                          $certname               = $facts['clientcert'],
  Puppet_enterprise::Unixpath     $confdir                = $puppet_enterprise::params::puppetdb_confdir,
  String                          $database_name          = $puppet_enterprise::puppetdb_database_name,
  Optional[String]                $database_password      = $puppet_enterprise::puppetdb_database_password,
  Puppet_enterprise::Port         $database_port          = $puppet_enterprise::database_port,
  String                          $database_user          = $puppet_enterprise::puppetdb_database_user,
  Variant[Pattern[/^[0-9]+$/],Integer] $gc_interval       = $puppet_enterprise::params::puppetdb_gc_interval,
  Optional[String]                $node_purge_ttl         = undef,
  Puppet_enterprise::Puppetdb_ttl $node_ttl               = $puppet_enterprise::params::puppetdb_node_ttl,
  Puppet_enterprise::Puppetdb_ttl $report_ttl             = $puppet_enterprise::params::puppetdb_report_ttl,
  Variant[Puppet_enterprise::Ip, Puppet_enterprise::Localhost] $listen_address = $puppet_enterprise::params::plaintext_address,
  Puppet_enterprise::Port         $listen_port            = $puppet_enterprise::params::puppetdb_listen_port,
  Puppet_enterprise::Ip           $ssl_listen_address     = $puppet_enterprise::params::ssl_address,
  Puppet_enterprise::Port         $ssl_listen_port        = $puppet_enterprise::puppetdb_port,
  Puppet_enterprise::Unixpath     $localcacert            = $puppet_enterprise::params::localcacert,
  String                          $database_properties    = $puppet_enterprise::database_properties,
  Hash                            $java_args              = $puppet_enterprise::params::puppetdb_java_args,
  Boolean                         $enable_system_metrics_collection = $puppet_enterprise::enable_system_metrics_collection,
  $auto_configure_sync = undef, # deprecated
  Optional[
    Array[
      Struct[{
        host => String,
        port => Integer,
        sync_interval_minutes => Integer}]]] $sync_peers  = undef,
  Optional[Array[String]]         $sync_whitelist         = undef,
  Array[String]                   $sync_allowlist         = pe_pick($sync_whitelist, []),
  Boolean                         $syncing                = $puppet_enterprise::replicating,
  String                          $rbac_host              = $puppet_enterprise::console_host,
  String                          $rbac_port              = $puppet_enterprise::api_port,
  String                          $rbac_prefix            = $puppet_enterprise::rbac_url_prefix,
) inherits puppet_enterprise {
  puppet_enterprise::deprecated_parameter{ 'puppet_enterprise::profile::puppetdb::whitelisted_certnames': replacement => 'puppet_enterprise::profile::puppetdb::allowlisted_certnames' }
  puppet_enterprise::deprecated_parameter{ 'puppet_enterprise::profile::puppetdb::sync_whitelist': 
    replacement         => 'puppet_enterprise::profile::puppetdb::sync_allowlist',
    suppress_on_install => true,
  }

  $cert_allowlist_path = '/etc/puppetlabs/puppetdb/certificate-allowlist'

  if ($auto_configure_sync != undef) {
    warning("The \$auto_configure_sync parameter is deprecated and will be ignored. \
Please explicitly configure PuppetDB sync using the \$sync_peers and \
\$sync_allowlist parameters.")
  }

  $puppetdb_ssl_dir = $puppet_enterprise::params::puppetdb_ssl_dir
  if $puppet_enterprise::database_ssl and $puppet_enterprise::database_cert_auth {
    $puppetdb_jdbc_ssl_properties = pe_join(
      [$database_properties,
       "&sslkey=${puppetdb_ssl_dir}/${certname}.private_key.pk8",
       "&sslcert=${puppetdb_ssl_dir}/${certname}.cert.pem"])
  } else {
    $puppetdb_jdbc_ssl_properties = $database_properties
  }

  if $rbac_host {
    $rbac_url = "https://${rbac_host}:${rbac_port}${rbac_prefix}"
  } else {
    $rbac_url = undef
  }

  class { 'puppet_enterprise::puppetdb':
    certname            => $certname,
    cert_allowlist_path => $cert_allowlist_path,
    confdir             => $confdir,
    database_name       => $database_name,
    database_port       => $database_port,
    database_host       => $database_host,
    database_user       => $database_user,
    database_password   => $database_password,
    gc_interval         => $gc_interval,
    node_purge_ttl      => $node_purge_ttl,
    node_ttl            => $node_ttl,
    report_ttl          => $report_ttl,
    listen_address      => $listen_address,
    listen_port         => $listen_port,
    ssl_listen_address  => $ssl_listen_address,
    ssl_listen_port     => $ssl_listen_port,
    localcacert         => $localcacert,
    database_properties => $puppetdb_jdbc_ssl_properties,
    java_args           => $java_args,
    rbac_url            => $rbac_url,
    sync_peers          => $sync_peers,
    syncing             => $syncing,
  }

  puppet_enterprise::certs { 'pe-puppetdb' :
    certname => $certname,
    container => 'puppetdb',
    cert_dir => $puppetdb_ssl_dir,
    make_pk8_cert => true,
  }

  file { $cert_allowlist_path:
    ensure  => file,
    group   => 'pe-puppetdb',
    owner   => 'pe-puppetdb',
    mode    => '0640',
    require => Package['pe-puppetdb']
  }

  $certs = pe_union($allowlisted_certnames,
                    pe_union($sync_allowlist,
                             [$puppet_enterprise::console_host,
                              $puppet_enterprise::puppet_master_host, # for the orchestrator
                              $master_certname,
                              $certname ]))

  class { 'puppet_enterprise::certs::puppetdb_allowlist':
    cert_allowlist_path => $cert_allowlist_path,
    certnames           => $certs,
    notify              => Service['pe-puppetdb'],
  }

  # The system class is included on all infra nodes
  if $enable_system_metrics_collection {
    include puppet_enterprise::pe_system_metrics
  }
}
