# This class is for managing the configuration file for PuppetDB's Trapperkeeper
# jetty instance.
#
# @param certname [String] Name of the clients certificate
# @param cert_whitelist_path Optional[String] * Deprecated * Use cert_allowlist_path instead
# @param cert_allowlist_path [String] Path to the certificate-allowlist file
# @param confdir [String] The path to PuppetDB's confdir
# @param listen_address [String] The address which the database is listening on
#        for plain text connections
# @param listen_port [Integer] The port which PuppetDB Listens on for plain text connections
# @param localcacert [String] The path to the local CA certificate
# @param ssl_dir [String] The directory where Puppet's ssl directory
# @param ssl_listen_address [String] The address which the database is
#        listening on for SSL connections
# @param ssl_listen_port [Integer] The port which PuppetDB Listens on for SSL connections
# @param ssl_protocols [Array[String]] The list of SSL protocols to allow.
# @param cipher_suites [Optional[String]] An optional list of cipher suites to
#        be used by puppetdb during ssl communications. If left undef the JDK defaults
#        will be used.  This parameter existed before ssl_cipher_suites was added
#        to puppet_enterprise so we leave this parameter as a string instead of
#        an array.
# @param tk_jetty_max_threads [Integer] The maximum number of threads that
#        Trapperkeeper's Jetty server can spin up.
# @param tk_jetty_request-header_max_size [Integer] Sets the maximum size of an HTTP Request Header.
class puppet_enterprise::puppetdb::jetty_ini(
  Optional[String] $cert_whitelist_path = undef,
  $cert_allowlist_path              = $cert_whitelist_path,
  $certname                         = $puppet_enterprise::puppetdb::certname,
  $confdir                          = $puppet_enterprise::params::puppetdb_confdir,
  $listen_address                   = $puppet_enterprise::params::plaintext_address,
  $listen_port                      = $puppet_enterprise::params::puppetdb_listen_port,
  $localcacert                      = $puppet_enterprise::params::localcacert,
  $ssl_dir                          = $puppet_enterprise::params::puppetdb_ssl_dir,
  $ssl_listen_address               = $puppet_enterprise::params::ssl_address,
  $ssl_listen_port                  = $puppet_enterprise::params::puppetdb_ssl_listen_port,
  Array[String] $ssl_protocols      = $puppet_enterprise::ssl_protocols,
  Optional[String] $cipher_suites   = pe_join($puppet_enterprise::ssl_cipher_suites, ','),
  $tk_jetty_max_threads             = undef,
  $tk_jetty_request_header_max_size = 65536,
) inherits puppet_enterprise::params {
  include puppet_enterprise::packages
  include puppet_enterprise::puppetdb::jetty_ini::deprecations
  
  file { "${confdir}/jetty.ini":
    ensure  => present,
    owner   => 'pe-puppetdb',
    group   => 'pe-puppetdb',
    mode    => '0640',
    require => Package['pe-puppetdb']
  }

  #Set the defaults
  Pe_ini_setting {
    ensure  => present,
    path    => "${confdir}/jetty.ini",
    section => 'jetty',
    require => File["${confdir}/jetty.ini"]
  }

  if pe_empty($listen_port) {
    pe_ini_setting {'puppetdb_host':
      ensure  => absent,
      setting => 'host',
    }
    pe_ini_setting {'puppetdb_port':
      ensure  => absent,
      setting => 'port',
    }
  } else {
    pe_ini_setting {'puppetdb_host':
      setting => 'host',
      value   => $listen_address,
    }

    pe_ini_setting {'puppetdb_port':
      setting => 'port',
      value   => $listen_port,
    }
  }

  pe_ini_setting {'puppetdb_sslhost':
    setting => 'ssl-host',
    value   => $ssl_listen_address,
  }

  pe_ini_setting {'puppetdb_sslport':
    setting => 'ssl-port',
    value   => $ssl_listen_port,
  }

  $ciphers = empty($cipher_suites) ? {
    true => puppet_enterprise::ciphers($ssl_protocols, 'iana').join(','),
    false => $cipher_suites,
  }

  pe_ini_setting {'puppetdb_cipher_suites':
    ensure  => present,
    setting => 'cipher-suites',
    value   => $ciphers,
  }

  pe_ini_setting {'puppetdb_ssl_key':
    setting => 'ssl-key',
    value   => "${ssl_dir}/${certname}.private_key.pem",
  }

  pe_ini_setting {'puppetdb_ssl_cert':
    setting => 'ssl-cert',
    value   => "${ssl_dir}/${certname}.cert.pem",
  }

  pe_ini_setting {'puppetdb_ssl_ca_cert':
    setting => 'ssl-ca-cert',
    value   => $localcacert,
  }

  pe_ini_setting {'puppetdb_client_auth':
    setting => 'client-auth',
    value   => 'want',
  }

  pe_ini_setting { 'puppetdb_ssl_protocols':
    setting => 'ssl-protocols',
    value => pe_join($ssl_protocols, ','),
  }

  # delete whitelist setting to avoid conflict in PDB with allowlist
  pe_ini_setting { 'puppetdb-certificate-whitelist':
    section => 'puppetdb',
    setting => 'certificate-whitelist',
    ensure  => absent,
  }

  pe_ini_setting { 'puppetdb-certificate-allowlist':
    section => 'puppetdb',
    setting => 'certificate-allowlist',
    value   => $cert_allowlist_path,
    require => File[$cert_allowlist_path],
  }


  if $tk_jetty_max_threads != undef {
    pe_ini_setting {'puppetdb_max-threads':
      setting => 'max-threads',
      value   => $tk_jetty_max_threads,
    }
  }

  pe_ini_setting {'puppetdb_request_header_max_size':
    setting => 'request-header-max-size',
    value   => $tk_jetty_request_header_max_size,
  }
}
