require 'date'
require 'yaml'
require 'base64'

class PELicense
  class InvalidLicenseError < StandardError; end

  LICENSE_PATH = '/etc/puppetlabs/license.key'

  attr_reader :to, :nodes, :start_date, :end_date, :complimentary, :uuid

  def initialize(to, nodes, start_date, end_date, uuid = nil, complimentary=false)
    @to = to
    @nodes = nodes
    @start_date = start_date
    @end_date = end_date
    @complimentary = complimentary
    @uuid = uuid
  end

  def self.from_file(filename)
    hash = nil
    begin
      hash = YAML.load_file(filename) || raise(InvalidLicenseError, "Could not find license information in #{filename}")
    rescue Errno::ENOENT => e
      raise InvalidLicenseError, "Could not find license file: #{e}"
    rescue Psych::SyntaxError => e
      raise InvalidLicenseError, "Could not load license file: #{e}"
    end

    if ! valid_license_date?(hash['start'])
      raise InvalidLicenseError, "The start date (#{hash['start']}) in the license file is improper"
    end

    if ! valid_license_date?(hash['end'])
      raise InvalidLicenseError, "The end date (#{hash['end']}) in the license file is improper"
    end

    unless hash['nodes'].is_a?(Integer) and hash['nodes'] > 0
      raise InvalidLicenseError, "The node count (#{hash["nodes"]}) in the license is improper"
    end

    self.new(hash['to'], hash['nodes'], hash['start'], hash['end'], hash['uuid'])
  end

  def self.load_license_key
    if File.exists?(LICENSE_PATH)
      self.from_file(LICENSE_PATH)
    else
      self.complimentary_license
    end
  end

  def self.complimentary_license
    new('N/A', 10, nil, nil, nil, true)
  end

  def self.valid_license_date?(date)
    date.nil? or date.is_a? Date
  end

  def started?(date=Date.today)
    # If there's no start date, it's always valid
    start_date.nil? || (start_date <= date)
  end

  def expired?(date=Date.today)
    # If there's no end date, it's valid indefinitely
    end_date && (end_date < date)
  end

  def days_remaining(date=Date.today)
    if expired?
      0
    elsif end_date
      (end_date - date).to_i
    else
      # If there's no end date, it's valid indefinitely
      Float::INFINITY
    end
  end

  def days_exceeded(date=Date.today)
    if expired?
      (date - end_date).to_i
    else
      0
    end
  end

  def authorization_token
    if !complimentary? && !expired?
      encoded_to = Base64.strict_encode64(@to || "")
      "pe-license #{@uuid} #{encoded_to}"
    end
  end

  def complimentary?
    !! @complimentary
  end
end
