# frozen_string_literal: true

require 'bolt/config'
require 'bolt/inventory'
require 'bolt/outputter'
require 'pe_installer/config'

module PeInstaller

  # Facade for initializing and interacting with the Bolt API.
  #
  # Supports running tasks, and running plans, which are actions Bolt can
  # handle either via ssh or pcp transport (assuming the plans are limited
  # to interacting with nodes via tasks).
  #
  # Bolt analytics are suppressed by initializing with the NoopClient.
  class BoltInterface
    attr_accessor :analytics
    attr_accessor :config
    attr_accessor :inventory
    attr_accessor :options
    attr_accessor :pal
    attr_accessor :puppetdb_client

    # Create a Bolt::Project instance rooted at the given directory.
    #
    # NOTE: I'm stipulating that we don't want the tool's behavior to change
    # depending on what directory the user executes it from. So the default
    # project dir will be under ~/.puppetlabs. As long as we initialize that
    # directory with a bolt-project.yaml before loading BoltInterface.generate_project,
    # Bolt won't wander the filesystem looking for project dirs.
    #
    # However, default Bolt config should still be loaded normally (see
    # {PeInstaller::BoltInterface.generate_config}).
    #
    # @param dir [String] This should point to a directory with the bolt-project.yaml
    # we intend to initialize with.
    # @return [Bolt::Project]
    def self.generate_project(dir = PeInstaller::Config.default_project_directory)
      Bolt::Project.new({}, dir)
    end

    # Create a Bolt::Config using configuration from the given project,
    # overriding configuration as provided.
    #
    # This will load default Bolt configuration from user/etc settings:
    #
    # /etc/puppetlabs/bolt
    # ~/.puppetlabs/etc/bolt
    #
    # @param project [Bolt::Project]
    # @param overrides [Hash<String,Object>] will override any configuration
    # loaded from +project+ (or defaults)
    # @return [Bolt::Config]
    def self.generate_config(project: generate_project, overrides: {})
      Bolt::Config.from_project(project, overrides)
    end

    def self.generate_pal(config)
      Bolt::PAL.new(
        Bolt::Config::Modulepath.new(config.modulepath),
        config.hiera_config,
        config.project.resource_types,
        config.compile_concurrency,
        config.trusted_external,
        config.apply_settings,
        nil # project
      )
    end

    def self.generate_puppetdb_config(config)
      Bolt::PuppetDB::Config.new(config: config.puppetdb, project: config.project.path)
    end

    def self.generate_puppetdb_client(config)
      Bolt::PuppetDB::Client.new(config: generate_puppetdb_config(config))
    end

    def self.generate_plugins(config:, pal:)
      Bolt::Plugin.new(config, pal)
    end

    def self.generate_inventory(config, plugins)
      Bolt::Inventory.from_config(config, plugins)
    end

    def self.instance(config = nil, options = {})
      config ||= generate_config
      pal = generate_pal(config)
      puppetdb_client = generate_puppetdb_client(config)
      plugins = generate_plugins(
        config: config,
        pal: pal
      )

      new(
        config: config,
        options: options,
        pal: pal,
        puppetdb_client: puppetdb_client,
        inventory: generate_inventory(config, plugins)
      )
    end

    def initialize(config:, options:, pal:, puppetdb_client:, inventory:)
      self.config = config
      self.options = options
      self.pal = pal
      self.puppetdb_client = puppetdb_client
      self.inventory = inventory
      self.analytics = Bolt::Analytics::NoopClient.new
    end

    def outputter
      Bolt::Outputter.for_format(config.format, config.color, options[:verbose], config.trace, config.spinner)
    end

    def log_outputter
      Bolt::Outputter::Logger.new(options[:verbose], config.trace)
    end

    def executor
      executor = Bolt::Executor.new(config.concurrency, analytics, options[:noop])
      # Skip out::message entirely for json format
      executor.subscribe(outputter) if options.fetch(:format, 'human') == 'human'
      executor.subscribe(log_outputter)
    end

    def run_plan(name, parameters = {})
      plan_context = {
        plan_name: name,
        params: parameters,
        description: "Manage Enterprise Puppet running plan #{name}",
      }

      executor.start_plan(plan_context)
      result = pal.run_plan(name, parameters, executor, inventory, puppetdb_client)

      # If a non-bolt exception bubbles up the plan won't get finished
      executor.finish_plan(result)
      executor.shutdown

      outputter.print_plan_result(result)

      result
    rescue Bolt::Error => e
      outputter.fatal_error(e)
      raise e
    end

    def run_task(name, targets, parameters = {})
      results = nil
      outputter.print_head

      elapsed_time = Benchmark.realtime do
        results = pal.run_task(
          name,
          targets,
          parameters,
          executor,
          inventory,
          "Manage Enterprise Puppet running task #{name}"
        )
      end

      executor.shutdown

      outputter.print_summary(results, elapsed_time)

      results
    rescue Bolt::Error => e
      outputter.fatal_error(e)
      raise e
    end
  end
end
