# frozen_string_literal: true

require 'bolt/inventory'
require 'pe_installer/inventory/node_roles'

module PeInstaller

  # Errors raised when working with inventory.
  class InventoryError < PeInstaller::Error; end

  # Abstraction around a Bolt::Inventory to handle looking up
  # groups matching the expected PE Infrastructure roles:
  # master, database, compilers, etc.
  #
  # Can be initially seeded from a {PeInstaller::Inventory::NodeRoles}
  # object as a means of instantiating commandline settings for node
  # roles within the inventory.
  #
  # Wraps a Bolt::Inventory.
  class Inventory

    # Generate a {PeInstaller::Inventory} and pre-populate targets passed on
    # the command line into their correct role groups.
    #
    # @param bolt_inventory [Bolt::Inventory] we're loading into.
    # @param seed_roles [PeInstaller::Inventory::NodeRoles] node roles to add.
    # @return [PeInstaller::Inventory]
    def self.with_seed_roles(bolt_inventory, seed_roles)
      inventory = new(bolt_inventory)

      seed_roles.each do |role, nodes|
        Array(nodes).each do |name|
          inventory._push(
            role.to_s,
            {
              'uri' => name,
              'vars' => {
                'source' => seed_roles.source,
              },
            }
          )
        end
      end

      inventory
    end

    attr_accessor :bolt_inventory

    # @param bolt_inventory [Bolt::Inventory] canonical inventory source.
    def initialize(bolt_inventory)
      self.bolt_inventory = bolt_inventory
    end

    def infrastructure
      _lookup('infrastructure')
    end

    def masters
      _lookup('masters')
    end

    def databases
      _lookup('databases')
    end

    def replicas
      _lookup('replicas')
    end

    def compilers
      _lookup('compilers')
    end

    # @return [Hash] representation of groups and targets.
    def to_h
      {
        'infrastructure' => {
          'masters' => masters.map(&:detail),
          'databases' => databases.map(&:detail),
          'replicas' => replicas.map(&:detail),
          'compilers' => compilers.map(&:detail),
        },
      }
    end

    # @api private
    def _lookup(role)
      bolt_inventory.get_targets(role)
    end

    # @api private
    def _push(role, target_hash)
      target = Bolt::Target.from_hash(target_hash, bolt_inventory)
      bolt_inventory.add_to_group([target], role)
      target
    end
  end
end
