# frozen_string_literal: true

require 'forwardable'

module PeInstaller

  # Errors assigning or reconciling nodes to Puppet Enterprise infrastructure roles.
  class NodeRoleError < PeInstaller::Error; end

  class Inventory

    # Immutable structure holding PE node roles provided directly from
    # the commandline or a pe.conf file.
    #
    # Provides direct accessors for the primary infrastructure roles and
    # a few Hash functions for convenience.
    #
    # * {NodeRoles.masters}
    # * {NodeRoles.databases}
    # * {NodeRoles.replicas}
    # * {NodeRoles.compilers}
    #
    # All accessors return arrays for consistency, even though some roles must
    # be singular (master, for example). It is not the responsibility of this
    # class to validate such things. See {PeInstaller::Architecture} for that.
    class NodeRoles

      # Generate a NodeRoles instance first from commandline args,
      # then absorbing from pe.conf.
      #
      # This will raise errors according to {PeInstaller::Inventory::NodeRoles.absorb()}.
      #
      # @param config [PeInstaller::Config] a loaded config instance.
      def self.from_config(config)
        node_roles = from_arguments(config.arguments)
        if !config.loaded?
          logger = Logging.logger[self]
          logger.warn("#{config} hasn't loaded external files yet, so we can't absorb node roles from pe.conf.")
        end
        pe_conf_roles = from_pe_conf(config.pe_conf_hash, config.pe_conf_file)
        node_roles.absorb(pe_conf_roles)
      end

      # Generate a NodeRoles instance by extracting node role details
      # from the passed hash of commandline arguments.
      #
      # @param command_arguments [Hash] of Thor commandline arguments
      # @return [NodeRoles]
      def self.from_arguments(command_arguments)
        new(
          {
            masters: command_arguments['master'],
            databases: command_arguments['database'],
            replicas: command_arguments['replicas'],
            compilers: command_arguments['compilers'],
          },
          "#{PeInstaller.tool_name}-commandline"
        )
      end

      # Generates a NodeRoles based on the core node parameters in a pe.conf Hash.
      #
      # puppet_enterprise::puppet_master_host
      # puppet_enterprise::database_host
      #
      # @param pe_conf_hash [Hash] a parsed pe.conf Hash.
      # @return [NodeRoles]
      def self.from_pe_conf(pe_conf_hash, pe_conf_file)
        new(
          {
            masters: pe_conf_hash['puppet_enterprise::puppet_master_host'],
            databases: pe_conf_hash['puppet_enterprise::database_host'],
          },
          pe_conf_file
        )
      end

      # Nothing set.
      def self.empty
        new({})
      end

      attr_reader :_hash
      attr_reader :source
      attr_reader :masters
      attr_reader :databases
      attr_reader :replicas
      attr_reader :compilers

      # provide a few of the more useful hash functions
      extend Forwardable
      def_delegators :@_hash, :each, :map, :each_with_object, :select, :reject

      def initialize(node_hash, source = 'no-source-given')
        @_hash = {
          masters: Array(node_hash[:masters].dup).freeze,
          databases: Array(node_hash[:databases].dup).freeze,
          compilers: Array(node_hash[:compilers].dup).freeze,
          replicas: Array(node_hash[:replicas].dup).freeze,
        }.freeze
        @masters = @_hash[:masters]
        @databases = @_hash[:databases]
        @compilers = @_hash[:compilers]
        @replicas = @_hash[:replicas]
        @source = Array(source.dup).flatten.freeze
        @logger = Logging.logger[self]
      end

      # Pulls in fields from the given {NodeRoles} instance so long as
      # they do not conflict with our existing non-empty values.
      #
      # This allows us to pull in missing fields from pe.conf.
      #
      # @param other [PeInstaller::Inventory::NodeRoles] to absorb from.
      # @return [PeInstaller::Inventory::NodeRoles] a new instance with the updated data.
      # @raise [PeInstaller::NodeRoleError] if the +other+ would overwrite a non-empty
      # field.
      def absorb(other)
        if !other.is_a?(PeInstaller::Inventory::NodeRoles)
          raise(
            PeInstaller::NodeRoleError,
            "Expected a PeInstaller::Inventory::NodeRoles, got #{other.class}: #{other}"
          )
        end

        new_node_hash = other.each_with_object({}) do |(role, other_nodes), collector|
          other_is_empty = other_nodes.nil? || other_nodes.empty?

          our_nodes = _hash[role]
          we_have_a_value = !our_nodes.nil? && !our_nodes.empty?

          if other_is_empty
            collector[role] = our_nodes
          elsif our_nodes != other_nodes
            if we_have_a_value
              raise(PeInstaller::NodeRoleError, <<~ERR.split.join(' '))
                There is a #{role} role conflict: was set to '#{our_nodes.join(',')}'
                from #{source}, but the #{other.source} has a value of '#{other_nodes.join(',')}'.
              ERR
            end
            @logger.info("Setting #{role} to '#{other_nodes.join(',')}' based on #{other.source}")
            collector[role] = other_nodes
          end
        end

        PeInstaller::Inventory::NodeRoles.new(new_node_hash, [source, other.source])
      end

      # Ignoring source for equality test; it's just there for auditing in logs.
      def ==(other)
        _hash == other._hash
      end

      def hash
        _hash.hash
      end

      # @return [Hash] An unfrozen hash of data.
      def to_h
        {
          masters: masters.dup,
          databases: databases.dup,
          replicas: replicas.dup,
          compilers: compilers.dup,
          source: source.dup,
        }
      end
    end
  end
end
