# frozen_string_literal: true

require 'net/http'
require 'uri'
require 'openssl'
require 'puppet'
require 'json'

# EnterpriseTasks isn't defined elsewhere.
# rubocop:disable Style/ClassAndModuleChildren
module EnterpriseTasks
  # Provides methods for making Net::HTTPs requests.
  # Authorization is reliant on the services being contacted
  # recognizing this host's Puppet certificate as a valid
  # client.
  module Network
    def initialize_puppet
      unless Puppet.settings.app_defaults_initialized?
        Puppet.initialize_settings(['--libdir=/dev/null', '--factpath=/dev/null'])
        Puppet.settings.app_defaults_initialized?
      end
    end

    def ssldir
      initialize_puppet
      Puppet[:ssldir]
    end

    def certname
      initialize_puppet
      Puppet[:certname]
    end

    def key_path
      "#{ssldir}/private_keys/#{certname}.pem"
    end

    def cert_path
      "#{ssldir}/certs/#{certname}.pem"
    end

    def key
      k = File.read(key_path)
      OpenSSL::PKey::RSA.new(k)
    end

    def cert
      c = File.read(cert_path)
      OpenSSL::X509::Certificate.new(c)
    end

    def cacert_path
      "#{ssldir}/certs/ca.pem"
    end

    # Generates a connection object with ssl authorization based on the local
    # Puppet cert and ca cert.
    def connection(host, port)
      conn = Net::HTTP.new(host, port)
      conn.use_ssl = true
      conn.ca_file = cacert_path
      conn.key = key
      conn.cert = cert
      conn.verify_mode = OpenSSL::SSL::VERIFY_PEER
      conn
    end

    # Makes a request against a #{connection} instance.
    # @return Net::HTTPResponse
    def request(host, port, endpoint, method: :get, body: nil)
      conn = connection(host, port)

      uri = URI("https://#{host}:#{port}/#{endpoint}")
      request = case method.to_s.downcase.to_sym
      when :get then Net::HTTP::Get.new(uri)
      when :put then Net::HTTP::Put.new(uri)
      when :post then Net::HTTP::Post.new(uri)
      when :delete then Net::HTTP::Delete.new(uri)
      end
      request.content_type = 'application/json'
      request.body = body.to_json if !body.nil?

      response = conn.request(request)

      case response
      when Net::HTTPSuccess, Net::HTTPRedirection
        return response
      else
        raise(
          EnterpriseTaskHelper::Error.new(
            "Failed Net::HTTPs request to #{host}:#{port}/#{endpoint}",
            'pe.enterprise-tasks-network/https-request-failed',
            {
              uri: request.uri.to_s,
              method: request.method,
              request_body: request.body,
              response_code: response.code,
              response_body: response.body,
            },
          ),
        )
      end
    end
  end
end
