# frozen_string_literal: true

require 'json'

# EnterpriseTasks isn't defined elsewhere.
# rubocop:disable Style/ClassAndModuleChildren
module EnterpriseTasks
  # Methods for interacting with a Puppet Enterprise RBAC system for
  # generating users and roles.
  #
  # Requires enterprise_tasks/network to be included in the Task as well.
  module RBAC

    # The name of the role used to grant temporary permissions to for enterprise_tasks functions.
    ENTERPRISE_TASKS_ROLE_NAME = 'Puppet Enterprise Tasks Role'
    # The login used by enterprise_tasks functions for puppet-access.
    ENTERPRISE_TASKS_USER_LOGIN = 'pe-enterprise-tasks-user'

    # RBAC endpoints lookup.
    ENDPOINTS = {
      roles: 'rbac-api/v1/roles',
      users: 'rbac-api/v1/users',
    }.freeze

    def rbac_request(path, body: nil, method: :get)
      endpoint = case path
      when Symbol then endpoint(path)
      else path
      end
      response = request(certname, 4433, endpoint, body: body, method: method)
      (response.body.nil? || response.body.empty?) ?
        response :
        JSON.parse(response.body)
    end

    # GET /roles
    # @return [Array] of RBAC role hashes.
    def roles
      rbac_request(:roles)
    end

    # GET /users
    # @return [Array] of RBAC user hashes.
    def users
      rbac_request(:users)
    end

    def _find(list, **kwargs)
      list.find do |i|
        kwargs.all? { |k, v| i[k.to_s] == v }
      end
    end

    # @return [Hash] the first Role hash matching the given keywords.
    def find_role(**kwargs)
      _find(roles, kwargs)
    end

    # @return [Hash] the first User hash matching the given keywords.
    def find_user(**kwargs)
      _find(users, kwargs)
    end

    # GET /<type>/<id>
    # @return [Hash] the RBAC hash for the type of object and the given id.
    def get(type, id)
      rbac_request("#{endpoint(type)}/#{id}")
    end

    # POST /roles
    # @param attrs [Hash] hash of RBAC Role attributes.
    # @return [String] endpoint for the newly created role.
    def add_role(attrs = {})
      role = attrs.each_with_object({}) do |pair, hash|
        k, v = pair
        hash[k.to_s] = v
      end
      role['permissions'] ||= []
      role['user_ids']    ||= []
      role['group_ids']   ||= []
      _add(:roles, role)
    end

    # POST /users
    # @return [String] endpoint for the newly craeted user.
    def add_user(attrs = {})
      user = attrs.each_with_object({}) do |pair, hash|
        k, v = pair
        hash[k.to_s] = v
      end

      user['role_ids'] ||= []
      _add(:users, user)
    end

    def _add(type, object)
      response = rbac_request(type, method: :post, body: object)
      location = response.header['location']
      get(type, location.split('/').last)
    end

    # PUT /<type>/<id>
    # Put request to update an object of the given type.
    # @param type [Symbol] must be one of the ENDPOINTS.keys
    # @param object [Hash] correct RBAC object hash for the given type.
    # @return [Hash] the modified object as returned by the server.
    def update(type, object)
      rbac_request("#{endpoint(type)}/#{object['id']}", method: :put, body: object)
    end

    # DELETE /<type>/<id>
    # @param type [Symbol] must be one of the ENDPOINTS.keys
    # @param object [Hash] correct RBAC object hash for the given type.
    # @return [NetHTTPResponse] success response object from Net::HTTP.
    def delete(type, object)
      rbac_request("#{endpoint(type)}/#{object['id']}", method: :delete)
    end

    def endpoint(type)
      ENDPOINTS[type] || raise(
        EnterpriseTaskHelper::Error.new(
          "Unknown RBAC endpoint #{type}.",
          'pe.enterprise-tasks-rbac/unknown-endpoint',
          {
            'requested' => type,
            'available' => ENDPOINTS,
          },
        ),
      )
    end

    # @return [Hash] a syntactically correct RBAC Permission hash for inclusion
    #   in a roles['permissions'] body.
    def permission(object_type, action, instance = '*')
      {
        'object_type' => object_type,
        'action'      => action,
        'instance'    => instance,
      }
    end

    # Find and clean up the pe-enterprise-tasks-user user and role.
    def delete_enterprise_tasks_user_and_role
      role = find_role(display_name: ENTERPRISE_TASKS_ROLE_NAME)
      user = find_user(login: ENTERPRISE_TASKS_USER_LOGIN)
      delete(:users, user) if user
      delete(:roles, role) if role
    end
  end
end
