# This plan is run over the orchestrator.  When regenerating a cert that contains
# subject alt names (defined in puppet.conf), those subject alt names MUST be
# passed in to the dns_alt_names parameter.  This is a security measure to ensure
# that unexpected alt names do not appear in the cert after regeneration.
#
# Steps the plan will take:
# 1. If we are regenerating the cert on a PE Compiler (compiler with PuppetDB),
#    then verify that pp_auth_role: pe_compiler appears in the
#    extension_requests in csr_attributes.yaml.
# 2. Stop all PE services on the node.
# 3. Run the regenerate_agent_certificate plan, (see: puppet infra run
#    regenerate_agent_certificate --help, for what that entails) with clean_crl=
#    true. 
plan enterprise_tasks::secondary_cert_regen(
  TargetSpec $target,
  TargetSpec $primary                = 'localhost',
  Optional[Hash] $custom_attributes  = undef,
  Optional[String] $dns_alt_names    = undef,
  Optional[Hash] $extension_requests = undef,
  Optional[Boolean] $force           = false,
) {
  enterprise_tasks::verify_node($primary, 'primary', $force)

  $role = enterprise_tasks::get_node_role($target)
  unless ['legacy_compiler', 'pe_compiler', 'replica'].any |$r| { $r == $role } {
    fail_plan('This node is not a secondary (compiler/replica) infrastructure node.')
  }

  $result_or_error = catch_errors() || {
    if $role == 'pe_compiler' {
      run_task(enterprise_tasks::ensure_compiler_csr_attributes, $target)
    }
    run_task(enterprise_tasks::pe_services, $target,
      role  => $role,
      state => 'stopped',
    )

    run_plan(enterprise_tasks::agent_cert_regen,
      primary            => $primary,
      agent              => $target,
      node_type          => $role,
      extension_requests => $extension_requests,
      custom_attributes  => $custom_attributes,
      dns_alt_names      => $dns_alt_names,
      clean_crl          => true,
      force              => $force,
    )
  }
  if $result_or_error =~ Error {
    return fail_plan($result_or_error)
  } else {
    return $result_or_error
  }
}
