# Orchestrates a fresh installation of all PE infrastructure.
#
# This includes the initial primary and primary/database installation, along with
# subsequent replica, compilers, load balancers, razor and agents.
#
# The plan assumes that code manager will be setup, so a control repo is required.
# An enterprise_tasks RBAC user is temporarily set up to allow the initial code deploy.
#
# TODO:
#  * load balancers
#  * load balanced agents
#  * razor
#
# @param primary
#   The primary PE node.
# @param puppetdb
#   The puppetdb node if different from primary (legacy split).
# @param database
#   The database node if different from the primary or puppetdb node.
# @param console
#   The console node if different from primary (legacy split).
# @param replicas
#   The replica node(s) to install.
# @param compilers
#   The compiler node(s) to install.
# @param compiler_sans
#   List of subject alt names for compiler certs.
# @param agents
#   Additional agent node(s) to install.
# @param version
#   A version string indicating either the family line (2019.1, 2019.2, etc.)
#   or an exact version of PE to download and install. If just the family is
#   given, the latest development build from that line will be installed.
# @param tarball
#   Alternately you may supply an absolute path to a PE tarball on the localhost
#   which you wish installed.
# @param other_pe_conf_parameters
#   A hash of additional PE module parameters to be added to pe.conf prior to
#   installation.
# @param control_repo_url
#   The url to the control repository (remote or local).
# @param control_repo_private_key_path
#   If the repo requires an ssh key to reach, this is the absolute path to the
#   where the private key will be on the primary.
# @param local_path_to_private_key
#   If you want a local private key copied into the $control_repo_private_key_path,
#   set it here.
# @param console_admin_password
#   The password for the admin account. Only used if < 2019.0.
# @param replication_timeout
#   The number of seconds to allow for replication provisioning to complete.
# @param enable_replica
#   Whether or not to enable the replica after provisioning succeeds.
# @param replica_topology
#   Which replica topology to configure.
# @param use_tempdirs
#   If true, generate proper tempdirs as workdirs for uploading PE and pe.conf files.
#   Otherwise use /root. Defaults to true. Set to false for manual testing where you
#   want the simplicity of using /root as the workdir.
# @param legacy_installer
#   Whether or not install will use the legacy puppet-enterprise-installer script.
#   Normally defaults to a version test unless set explicitly.
plan enterprise_tasks::testing::install(
  Variant[Target,String] $primary,
  Variant[Target,String] $puppetdb = $primary,
  Variant[Target,String] $database = $puppetdb,
  Variant[Target,String] $console = $primary,
  TargetSpec $replicas = [],
  TargetSpec $compilers = [],
  Optional[String] $compiler_sans = 'puppet',
  TargetSpec $agents = [],
  String $control_repo_url = constants()['default_control_repo_url'],
  Boolean $enable_replica = true,
  Integer $replication_timeout = 1800,
  Enum['mono','mono-with-compile'] $replica_topology = 'mono',
  Boolean $use_tempdirs = true,
  Optional[Enterprise_tasks::Pe_version_or_family] $version = undef,
  Optional[Enterprise_tasks::Absolute_path] $tarball = undef,
  Optional[Hash] $other_pe_conf_parameters = {},
  Optional[Enterprise_tasks::Absolute_path] $control_repo_private_key_path = undef,
  Optional[Enterprise_tasks::Absolute_path] $local_path_to_private_key = undef,
  Optional[Variant[Sensitive[String],String]] $console_admin_password = undef,
  Optional[Boolean] $legacy_installer = undef,
) {
  $core_nodes = [$primary, $puppetdb, $database, $console].unique()
  if ($core_nodes.size() > 1 and !empty($replicas)) {
    fail_plan("Currently, replica can only be installed alongside a monolithic primary; however, the following core nodes have been assigned: ${core_nodes}.")
  }

  if ($version == undef and $tarball == undef) or ($version != undef and $tarball != undef) {
    fail_plan("You must specify either version or tarball, but not both. Received version: '${version}' and tarball: '${tarball}'")
  }

  enterprise_tasks::message('install', 'Installing the primary PE infrastructure.')
  $_pe_version = enterprise_tasks::lookup_pe_version($version)
  $_pe_family = enterprise_tasks::first_defined($_pe_version, $tarball).match(/\d{4}\.\d+/)[0]
  $primary_target = get_targets($primary)[0]

  # We can and should skip adding the admin password to pe.conf unless we are
  # installing < 2019.0
  if (versioncmp($_pe_family, '2019.0') == -1) {
    if $console_admin_password =~ Undef {
      enterprise_tasks::message('install', "Since ${_pe_family} requires a console_admin_password set in pe.conf, and no console_admin_password was given, we will be generating one randomly.")
      $_console_admin_password_for_pe_conf = enterprise_tasks::generate_random_password(16)
    } else {
      $_console_admin_password_for_pe_conf = $console_admin_password
    }
    enterprise_tasks::set_feature($primary_target, '2018.1.x', true)
  } else {
    $_console_admin_password_for_pe_conf = undef
  }

  if ($legacy_installer or (versioncmp($_pe_family, '2019.8') == -1)) {
    $_legacy_installer = true
  } else {
    $_legacy_installer = false
  }
  run_plan('enterprise_tasks::testing::install_pe',
    'primary'                  => $primary,
    'puppetdb'                 => $puppetdb,
    'database'                 => $database,
    'console'                  => $console,
    'version'                  => $_pe_version,
    'tarball'                  => $tarball,
    'console_admin_password'   => $_console_admin_password_for_pe_conf,
    'other_pe_conf_parameters' => $other_pe_conf_parameters,
    'use_tempdirs'             => $use_tempdirs,
    'legacy_installer'         => $_legacy_installer,
  )

  if (versioncmp($_pe_family, '2018.1') == 1) and $console_admin_password =~ NotUndef {
    enterprise_tasks::message('install', 'Set RBAC admin account password for console access')
    $bootstrap_password_result = run_task('enterprise_tasks::bootstrap_console_admin_password',
      $primary,
      'password'      => $console_admin_password,
      '_catch_errors' =>  true,
    ).first()
    if !$bootstrap_password_result.ok() {
      # If we are re-running the plan, this will fail, so just warn and move on.
      warning("Failed to set console admin password: ${bootstrap_password_result.value()}\nThis is unusual for a fresh installation, but could occur if the plan was re-run.")
    }
  }

  enterprise_tasks::message('install', 'Setting up code manager.')
  run_plan('enterprise_tasks::testing::setup_code_manager',
    'primary'                       => $primary,
    'control_repo_url'              => $control_repo_url,
    'control_repo_private_key_path' => $control_repo_private_key_path,
    'local_path_to_private_key'     => $local_path_to_private_key,
  )

  if !empty($replicas) {
    enterprise_tasks::message('install', 'Installing replica.')
    run_plan('enterprise_tasks::testing::install_replica',
      'replicas'            => $replicas,
      'primary'             => $primary,
      'replication_timeout' => $replication_timeout,
      'enable'              => $enable_replica,
      'topology'            => $replica_topology,
    )
  }

  if !empty($compilers) {
    enterprise_tasks::message('install', 'Installing compilers.')
    run_plan('enterprise_tasks::testing::install_compilers',
      'compilers'         => $compilers,
      'primary'           => $primary,
      'subject_alt_names' => $compiler_sans,
    )
  }

  if !empty($agents) {
    enterprise_tasks::message('install', 'Installing agents.')
    run_plan('enterprise_tasks::testing::install_agents',
      'agents'  => $agents,
      'primary' => $primary,
    )
  }
}
