# Install puppet-agent from the given primary node onto one or more agent nodes
# using the boostrap module (which uses pe_repo's install scripts).
#
# Sign the certificates.
# 
# Run puppet on the nodes to complete the installation.
#
# @param $agents [TargetSpec] The nodes to install puppet-agent on.
# @param $primary [Variant[Target,String]] The primary node to install from and to sign from.
# @param $subject_alt_names [Optional[String]] Additional DNS names to be added to the cert.
# @param $custom_attributes [Variant[Hash,Array[Pattern[/\w+=\w+/]]]] Hash or
#   an Array of attributes destined for the CSR via Puppet's csr_attributes.yaml file.
# @param $extension_requests [Variant[Hash,Array[Pattern[/\w+=\w+/]]]] hash or
#   an Array of attributes to be added as extensions in the final Certificate via
#   Puppet's csr_attributes.yaml file.
plan enterprise_tasks::testing::install_agents(
  TargetSpec $agents,
  Variant[Target,String] $primary,
  Variant[Hash,Array[Pattern[/\w+=\w+/]]] $custom_attributes = [],
  Variant[Hash,Array[Pattern[/\w+=\w+/]]] $extension_requests = [],
  Optional[String] $subject_alt_names = undef,
) {
  $constants = constants()

  $agent_targets = get_targets($agents)
  $primary_target = get_targets($primary)[0]

  enterprise_tasks::message('install_agents', 'Checking connectivity to primary and agent nodes.')
  enterprise_tasks::test_connection($agent_targets + $primary_target)

  case $primary {
    'localhost': {
      $primary_hostname = run_command(@(EOS/L), $primary).first['stdout'].strip
        puppet config print certname --section agent
      |-EOS
      $bootstrap_module = 'pe_bootstrap'
      $bootstrap_primary_arg = 'server'
    }
    default: {
      $primary_hostname = $primary_target.host()
      $bootstrap_module = 'bootstrap'
      $bootstrap_primary_arg = 'master'
    }
  }

  $cacert = run_command("cat ${constants['ca_pem']}", $primary_target).first['stdout']

  $_bootstrap_args = {
    $bootstrap_primary_arg => $primary_target.host(),
    'cacert_content'    => $cacert,
    'extension_request' => enterprise_tasks::convert_agent_installer_arg($extension_requests),
    'custom_attribute'  => enterprise_tasks::convert_agent_installer_arg($custom_attributes),
    '_catch_errors'     => true,
  }
  if $subject_alt_names =~ Undef {
    $bootstrap_args = $_bootstrap_args
    $allow_sans = false
  } else {
    $bootstrap_args = $_bootstrap_args + {
      'dns_alt_names' => $subject_alt_names
    }
    $allow_sans = true
  }

  $install_results = run_task($bootstrap_module, $agent_targets, $bootstrap_args)
  $install_failures = $install_results.ok() ? {
    true    => [],
    default => $install_results.error_set(),
  }

  $failed_installs = $install_failures.map() |$f| { $f.target() }
  $successful_installs = $agent_targets - $failed_installs

  $successful_installs.each |$n| {
    enterprise_tasks::message('install_agents', "Signing ${n}")
    run_task(enterprise_tasks::puppetserver_ca_sign, $primary_target,
      'host'                    => $n.host(),
      'allow_subject_alt_names' => $allow_sans,
    )
  }
  # Disabling the puppet service greatly speeds up the subsequent puppet run because we
  # aren't waiting for the initial ssl_lock held by the bootstrapping puppet daemon to
  # run through it's default waitforcert 120s.
  run_task(enterprise_tasks::pe_services, $successful_installs, state => 'stopped')
  # Once agent service handling is plumbed back into
  # puppet_enterprise::profile::agent from the next branch, then service status should
  # be automaticlly enforced correctly based on how we installed PE.
  run_task(enterprise_tasks::run_puppet, $successful_installs, 'max_timeout' => 256)

  enterprise_tasks::message('install_agents', 'Validating that agents nodes are steady state')
  run_task(enterprise_tasks::run_puppet, $successful_installs, 'exit_codes' =>  [0])

  if !$install_results.ok() {
    return $install_results.error_set()
  }
}
