# Install replica.
#
# Assumes a monolithic instance of PE is installed, and that code manager has
# been setup.
#
# @param $replicas [TargetSpec] the nodes to install replica on.
# @param $primary [Variant[Target,String]] the primary node.
# @param replication_timeout [Integer] number of seconds to allow for
#   replication provisioning to complete.
# @param streaming [Boolean] whether or not to use the faster streaming
#   option when replicating.
# @param enable [Boolean] whether or not to enable the replica after
#   provisioning succeeds.
# @param topology [Enum['mono','mono-with-compile']] which replica topology to
#   configure.
# @param rbac_account [Optional[String]] the account to provision and enable as.
#   A temporary account will be generated if not given.
# @param rbac_password [Optional[String]] the password for the $rbac_account.
plan enterprise_tasks::testing::install_replica(
  TargetSpec $replicas,
  Variant[Target,String] $primary,
  Integer $replication_timeout = 3600,
  Boolean $streaming = true,
  Boolean $enable = true,
  Enum['mono','mono-with-compile'] $topology = 'mono',
  Optional[String] $rbac_account = undef,
  Optional[Sensitive[String]] $rbac_password = undef,
) {

  $replica_targets = get_targets($replicas)
  $primary_target = get_targets($primary)[0]

  enterprise_tasks::message('install_replica', 'Checking connectivity to infrastructure nodes.')
  enterprise_tasks::test_connection($primary_target)
  enterprise_tasks::test_connection($replica_targets)

  run_plan('enterprise_tasks::testing::install_agents',
    'agents'     => $replica_targets,
    'primary'    => $primary_target,
  )

  get_targets($replica_targets).each |$replica| {
    enterprise_tasks::with_puppet_access($primary_target,
      'account'    => $rbac_account,
      'password'   => $rbac_password,
      'lifetime'   => "${replication_timeout}s",
    ) |$token_file, $_token| {
      enterprise_tasks::message('install_replica', "Provisioning ${$replica.host()}")
      run_task('enterprise_tasks::provision_replica', $primary_target,
        'host'                => $replica.host(),
        'replication_timeout' => $replication_timeout,
        'streaming'           => $streaming,
        'token_file'          => $token_file,
      )

      if $enable {
        enterprise_tasks::message('install_replica', "Enabling ${$replica.host()}")
        run_task('enterprise_tasks::enable_replica', $primary_target,
          'host'       => $replica.host(),
          'topology'   => $topology,
          'token_file' => $token_file,
        )
      }
    }
  }
}
