# Note: This plan requires user_data.conf to be present. We can't run
# it from inside this plan since the plan is meant to be run AFTER
# the hostname has already changed and therefore
# puppet infra recover_configuration will not work correctly.
plan enterprise_tasks::testing::refresh_primary_hostname(
  Optional[TargetSpec] $primary              = 'localhost',
  Optional[String]     $old_primary_certname = undef,
) {
  $puppet_bin = constants()['puppet_bin']
  $primary_target = get_targets($primary)[0]
  run_plan('facts', 'targets' => $primary_target)
  $primary_fqdn = $primary_target.facts()['fqdn']

  # Get old certname. This is overrideable in case the plan has issues part way
  # and puppet.conf has already been modified.
  $current_puppet_certname = run_command('/opt/puppetlabs/bin/puppet config print certname --section main', $primary).first.value['stdout'][0,-2]
  $prev_primary_certname = empty($old_primary_certname) ? {
    true  => $current_puppet_certname,
    false => $old_primary_certname,
  }

  if $primary_fqdn == $prev_primary_certname {
    enterprise_tasks::message('refresh_primary_hostname', "Current hostname ${primary_fqdn} has not changed. Exiting.")
    return undef
  }
  enterprise_tasks::message('refresh_primary_hostname', "Changing primary hostname from ${prev_primary_certname} to ${primary_fqdn}...")

  $status_hash = run_plan(enterprise_tasks::get_service_status, target => $primary,
    service    => 'puppet')

  ### Disabled until we are ready to make this user-facing ###
  # Replicas and compilers both have the 'master' role, external postgres has the database role,
  # so this should cover all of the infra nodes. 
  #$infra_nodes = enterprise_tasks::get_nodes_with_profile('master') + enterprise_tasks::get_nodes_with_profile('database')
  #$infra_non_primary = $infra_nodes.flatten.unique.filter |$node| { $node != $primary_fqdn and $node != $current_puppet_certname }
  $result_or_error = catch_errors() || {
    # Change this to [$primary] + $infra_non_primary when we make this user-facing
    enterprise_tasks::with_agent_disabled($primary) || {
      enterprise_tasks::message('refresh_primary_hostname', 'Stopping PE services...')
      run_task(enterprise_tasks::pe_services, $primary,
        role  => 'primary',
        state => 'stopped',
      )

      enterprise_tasks::message('refresh_primary_hostname', 'Modifying configuration files to use new hostname...')
      run_command("sed -i 's/${prev_primary_certname}/${primary_fqdn}/g' /etc/puppetlabs/puppet/puppet.conf", $primary)
      run_command("sed -i 's/${prev_primary_certname}/${primary_fqdn}/g' /etc/puppetlabs/enterprise/conf.d/pe.conf", $primary)
      # Because we have to set up the pdbpreload VM using the trusted certname fact for puppet_master_host,
      # and having puppet_master_host set to the trusted fact causes problems when pe.conf gets copied elsewhere,
      # we replace instances of the trusted fact as well. May not want this if we make the plan user-facing.
      run_command("sed -i 's/%{::trusted.certname}/${primary_fqdn}/g' /etc/puppetlabs/enterprise/conf.d/pe.conf", $primary)
      run_command("sed -i 's/${prev_primary_certname}/${primary_fqdn}/g' /etc/puppetlabs/enterprise/conf.d/user_data.conf", $primary)

      # These will get regenerated. We're clearing them out here because while puppet
      # will add configuration for the new certname, it will not remove the old certname
      run_command('echo > /etc/puppetlabs/nginx/conf.d/proxy.conf', $primary)
      run_command('echo > /etc/puppetlabs/nginx/conf.d/http_redirect.conf', $primary)
      run_command('echo > /etc/puppetlabs/puppetdb/certificate-whitelist', $primary)
      run_command('echo > /etc/puppetlabs/console-services/rbac-certificate-whitelist', $primary)

      enterprise_tasks::message('refresh_primary_hostname', 'Clearing cached catalogs...')
      run_command('rm -f /opt/puppetlabs/puppet/cache/client_data/catalog/*', $primary)

      ### Disabled until we are ready to make this user-facing ###
      #enterprise_tasks::message('refresh_primary_hostname', 'Modifying puppet.conf on infrastructure nodes...')
      # We have to modify puppet.conf here first because primary_cert_regen will run puppet 
      # and regen the cert on an external postgres node.
      #$infra_non_primary.each |$node| {
      #  run_command("sed -i 's/${prev_primary_certname}/${primary_fqdn}/g' /etc/puppetlabs/puppet/puppet.conf", $node)
      #}

      # We have to skip past verification since verify_node won't be able to talk to PDB
      # as it will still be pointing at the old certname
      run_plan(enterprise_tasks::primary_cert_regen, primary => $primary, force => true)

      # For some totally bizarre reason, on at least Ubuntu (not sure about other platforms),
      # about half the time the RBAC service in pe-console-services gets into a weird state
      # and just returns 500s when you try to do a node purge. Restarting the service
      # fixes it, instead of wasting more time trying to root cause this weird bug,
      # I'm just restarting it here.
      run_task(service, $primary,
              action        => 'restart',
              name          => 'pe-console-services')

      # This assumption really only works for pdbpreload when we know we don't have
      # a replica in the mix. Fix this up before making the plan public.
      apply($primary){
        pe_node_group { 'PE Infrastructure Agent':
          classes => {
            'puppet_enterprise::profile::agent' => {
              'manage_puppet_conf' => true,
              'master_uris'        => undef,
              'primary_uris'       => ["${primary_fqdn}:8140"],
              'pcp_broker_list'    => ["${primary_fqdn}:8142"],
              'server_list'        => ["${primary_fqdn}:8140"],
            }
          }
        }
      }
      # The puppet run in primary_cert_regen set server_list to localhost.localdomain so fix it here.
      run_command("${puppet_bin} config set --section agent server_list ${primary_fqdn}:8140", $primary)
      run_task(enterprise_tasks::run_puppet, $primary)

      enterprise_tasks::message('refresh_primary_hostname', 'Purging old primary certname and deleting its certificate...')
      run_command("${puppet_bin} node purge ${prev_primary_certname}", $primary)
      run_task(enterprise_tasks::delete_cert, $primary, certname => $prev_primary_certname)

      enterprise_tasks::message('refresh_primary_hostname', 'Unpinning old certname from PE node groups...')
      run_command("${puppet_bin} resource pe_node_group 'PE Certificate Authority' unpinned='${prev_primary_certname}'", $primary)
      run_command("${puppet_bin} resource pe_node_group 'PE Console' unpinned='${prev_primary_certname}'", $primary)
      run_command("${puppet_bin} resource pe_node_group 'PE Database' unpinned='${prev_primary_certname}'", $primary)
      run_command("${puppet_bin} resource pe_node_group 'PE Master' unpinned='${prev_primary_certname}'", $primary)
      run_command("${puppet_bin} resource pe_node_group 'PE Orchestrator' unpinned='${prev_primary_certname}'", $primary)
      run_command("${puppet_bin} resource pe_node_group 'PE PuppetDB' unpinned='${prev_primary_certname}'", $primary)
      $ha_master_group = run_command("${puppet_bin} resource pe_node_group 'PE HA Master'", $primary).first.value['stdout']
      if $ha_master_group =~ /present/ {
        run_command("${puppet_bin} resource pe_node_group 'PE HA Master' unpinned='${prev_primary_certname}'")
      }

      enterprise_tasks::message('refresh_primary_hostname', 'Running puppet to populate services.conf with new certname...')
      run_task(enterprise_tasks::run_puppet, $primary)

      ### Disabled until we are ready to make this user-facing ###
      #enterprise_tasks::message('refresh_primary_hostname', 'Running puppet on infrastructure nodes...')
      #$infra_non_primary.each |$node| {
      #  run_task(enterprise_tasks::run_puppet, $node, max_timeout => 256)
      #}
    }
  }
  enterprise_tasks::message('refresh_primary_hostname', 'Applying original puppet service state...')
  run_command("${puppet_bin} resource service puppet ensure=${status_hash[status]} enable=${status_hash[enabled]}", $primary)
  if $result_or_error =~ Error {
    fail_plan($result_or_error)
  }
}
